#!/usr/bin/env python
#
# This file is part of vospace-transfer-service
# Copyright (C) 2021 Istituto Nazionale di Astrofisica
# SPDX-License-Identifier: GPL-3.0-or-later
#

import logging
import os

from redis_log_handler import RedisLogHandler
from redis_rpc_server import RedisRPCServer
from config import Config
from db_connector import DbConnector


class JobRPCServer(RedisRPCServer):

    def __init__(self, host, port, db, rpcQueue):
        self.type = "job"
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(params["user"],
                                  params["password"],
                                  params["host"],
                                  params.getint("port"),
                                  params["db"],
                                  1,
                                  2,
                                  self.logger)
        super(JobRPCServer, self).__init__(host, port, db, rpcQueue)

    def callback(self, requestBody):
        # 'requestType' attribute is mandatory
        if "requestType" not in requestBody:
            errorMsg = "Malformed request, missing parameters."
            self.logger.error(errorMsg)
            response = { "responseType": "ERROR",
                         "errorCode": 1,
                         "errorMsg": errorMsg }
        elif requestBody["requestType"] == "JOB_LIST":
            try:
                result = self.dbConn.getActiveJobs()
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response
            else:
                response = { "responseType": "LST_DONE", 
                             "jobList": result }
        elif requestBody["requestType"] == "JOB_BY_PHASE":
            jobPhase = requestBody["jobPhase"]
            try:
                result = self.dbConn.getJobsByPhase(jobPhase)
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response
            else:
                response = { "responseType": "LST_BY_PHASE_DONE", 
                             "jobList": result }
        elif requestBody["requestType"] == "JOB_INFO":
            jobId = requestBody["jobId"]
            try:
                if self.dbConn.jobExists(jobId):
                    result = self.dbConn.getJobInfo(jobId)
                    response = { "responseType": "LST_INFO_DONE", 
                                 "jobInfo": result }
                else:
                    errorMsg = "Wrong job ID."
                    self.logger.error(errorMsg)
                    response = { "responseType": "ERROR",
                                 "errorCode": 3,
                                 "errorMsg": errorMsg }
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response                
        elif requestBody["requestType"] == "JOB_RESULTS":
            jobId = requestBody["jobId"]
            try:
                if self.dbConn.jobExists(jobId):
                    result = self.dbConn.getJobResults(jobId)
                    response = { "responseType": "LST_RESULTS_DONE", 
                                 "jobResults": result }
                else:
                    errorMsg = "Wrong job ID."
                    self.logger.error(errorMsg)
                    response = { "responseType": "ERROR",
                                 "errorCode": 3,
                                 "errorMsg": errorMsg }
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response
        elif requestBody["requestType"] == "JOB_SEARCH":
            searchStr = requestBody["searchStr"]
            try:
                result = self.dbConn.searchJobs(searchStr)
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response
            else:
                response = { "responseType": "SEARCH_DONE", 
                             "jobSearch": result }
        else:
            errorMsg = "Unkown request type."
            self.logger.error(errorMsg)
            response = { "responseType": "ERROR",
                         "errorCode": 4,
                         "errorMsg": errrorMsg }
        return response

    def run(self):
        self.logger.info(f"Starting RPC server of type {self.type}...")
        super(JobRPCServer, self).run()
