/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace.persistence;

import it.inaf.oats.vospace.datamodel.collections.NodeCollection;
import java.util.List;
import java.util.Optional;
import javax.sql.DataSource;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.context.junit.jupiter.SpringExtension;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.util.ReflectionTestUtils;

/**
 *
 * @author Nicola Fulvio Calabria <nicola.calabria at inaf.it>
 */
@ExtendWith(SpringExtension.class)
@ContextConfiguration(classes = {DataSourceConfig.class})
@TestPropertySource(locations = "classpath:test.properties")
public class CollectionsDAOTest {

    private static final String AUTHORITY = "example.com!vospace";

    private CollectionsDAO collectionsDAO;
    private NodeDAO nodeDAO;

    @Autowired
    private DataSource dataSource;

    @BeforeEach
    public void init() {
        collectionsDAO = new CollectionsDAO(dataSource);
        nodeDAO = new NodeDAO(dataSource);
        ReflectionTestUtils.setField(nodeDAO, "authority", AUTHORITY);
    }

    @Test
    public void testInsertAndDeleteCollection() {
        assertTrue(collectionsDAO.getUserNodeCollections("pippo1").isEmpty());

        collectionsDAO.createNewCollection("collection1", "pippo1");
        collectionsDAO.createNewCollection("collection2", "pippo1");

        List<NodeCollection> ncl
                = collectionsDAO.getUserNodeCollections("pippo1");

        assertEquals(2, ncl.size());

        for (NodeCollection nc : ncl) {
            collectionsDAO.deleteCollection(
                    nc.getId(), "pippo1"
            );
        }

        assertTrue(collectionsDAO.getUserNodeCollections("pippo1").isEmpty());
    }

    @Test
    public void testAddAndRemoveNode() {
        List<NodeCollection> collections = 
                collectionsDAO.getUserNodeCollections("pippo");
        
        assertEquals(1, collections.size());
        
        NodeCollection nc = collections.get(0);
        
        Optional<Long> maybeNodeId = nodeDAO.getNodeId("/test1/f1/f2_renamed");
        assertTrue(maybeNodeId.isPresent());
        Long nodeId = maybeNodeId.get();
        
        collectionsDAO.addNodeToCollection(nodeId, nc.getId());
        
        // Add some logic for checks
        
        collectionsDAO.removeNodeFromCollection(nodeId, nc.getId());       
    }

}
