/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace.persistence;

import it.inaf.oats.vospace.datamodel.collections.NodeCollection;
import it.inaf.oats.vospace.datamodel.collections.NodeDetails;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Optional;
import javax.sql.DataSource;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Repository;

/**
 *
 * @author Nicola Fulvio Calabria <nicola.calabria at inaf.it>
 */
@Repository
public class CollectionsDAO {

    private final JdbcTemplate jdbcTemplate;

    @Autowired
    public CollectionsDAO(DataSource dataSource) {
        jdbcTemplate = new JdbcTemplate(dataSource);
    }

    public void createNewCollection(String title, String ownerId) {

        String sql = "INSERT INTO collections (title, owner_id) VALUES (?,?)";

        jdbcTemplate.update(conn -> {
            PreparedStatement ps = conn.prepareStatement(sql);
            int i = 0;
            ps.setString(++i, title);
            ps.setString(++i, ownerId);
            return ps;
        });
    }

    public Optional<NodeCollection> getNodeCollectionById(Long id) {
        String sql = "SELECT collection_id, title, owner_id FROM collections\n"
                + "WHERE collection_id = ?";

        Optional<NodeCollection> nc = jdbcTemplate.query(
                conn -> {
                    PreparedStatement ps = conn.prepareStatement(sql);
                    ps.setLong(1, id);
                    return ps;
                },
                rs -> {
                    if (!rs.next()) {
                        return Optional.empty();
                    } else {
                        return Optional.of(getNodeCollectionFromResultset(rs));
                    }

                });

        return nc;

    }
    
    public List<NodeDetails> getNodeDetailsInCollection(Long collectionId) {
        String sql = "SELECT node_id, get_vos_path(node_id) as node_vos_path\n"
                + "FROM collections_node WHERE collection_id = ?";
        
        return jdbcTemplate.query(                
                conn-> {
                    PreparedStatement ps = conn.prepareStatement(sql);
                    ps.setLong(1, collectionId);
                    return ps;
                },                
                (row, index) -> {
                    return this.getNodeDetailsFromResultset(row);                    
                });
                
    }
    
    private NodeDetails getNodeDetailsFromResultset(ResultSet rs) 
            throws SQLException
    {
        NodeDetails nd = new NodeDetails();
        nd.setNodeId(rs.getLong("node_id"));
        nd.setNodeVosPath(rs.getString("node_vos_path"));
        
        return nd;
    }
    
    public List<NodeCollection> getUserNodeCollections(String userId) {
        String sql = "SELECT collection_id, title, owner_id FROM collections\n"
                + "WHERE owner_id = ?";

        List<NodeCollection> nc = jdbcTemplate.query(
                conn -> {
                    PreparedStatement ps = conn.prepareStatement(sql);
                    ps.setString(1, userId);
                    return ps;
                },
                (row, index) -> {
                    return this.getNodeCollectionFromResultset(row);
                });

        return nc;
    }
    
    public void addNodeToCollection(Long nodeId, Long collectionId) {
        String sql = "INSERT INTO collections_node (collection_id, node_id)\n"
                + "VALUES (?,?)";
        
        jdbcTemplate.update(conn -> {
            PreparedStatement ps = conn.prepareStatement(sql);
            int i = 0;
            ps.setLong(++i, collectionId);
            ps.setLong(++i, nodeId);
            return ps;
        });
    }
    
    public int removeNodeFromCollection(Long nodeId, Long collectionId) {
        String sql = "DELETE FROM collections_node WHERE collection_id = ? and node_id = ?";
        
        // Returns the number of deleted nodes: must be 1
        return jdbcTemplate.update(sql, collectionId, nodeId);
    }
    
    public int deleteCollection(Long collectionId, String userId) {       
        String sql  = "DELETE FROM collections WHERE collection_id = ? AND owner_id = ?";
        
        return jdbcTemplate.update(sql, collectionId, userId);
    }

    private NodeCollection getNodeCollectionFromResultset(ResultSet rs)
            throws SQLException {
        NodeCollection nc = new NodeCollection(
                rs.getLong("collection_id"),
                rs.getString("title"),
                rs.getString("owner_id")
        );

        return nc;
    }
}
