/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace;

import it.inaf.oats.vospace.persistence.CollectionsDAO;
import it.inaf.oats.vospace.datamodel.collections.NodeCollection;
import it.inaf.oats.vospace.datamodel.collections.NodeDetails;
import it.inaf.oats.vospace.exception.InvalidArgumentException;
import it.inaf.oats.vospace.exception.NodeBusyException;
import it.inaf.oats.vospace.exception.PermissionDeniedException;
import it.inaf.oats.vospace.exception.NodeNotFoundException;
import it.inaf.oats.vospace.persistence.NodeDAO;
import it.inaf.oats.vospace.persistence.NodeDAO.ShortNodeDescriptor;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.EnableTransactionManagement;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Transactional;

/**
 *
 * @author Nicola Fulvio Calabria <nicola.calabria at inaf.it>
 */
@Service
@EnableTransactionManagement
public class CollectionsService {

    @Autowired
    private CollectionsDAO collectionsDAO;

    @Autowired
    private NodeDAO nodeDAO;

    public List<NodeCollection> listCollections(String userId) {

        this.doCheckUserAuth(userId);

        List<NodeCollection> result = new ArrayList<>();
        result.addAll(collectionsDAO.getUserNodeCollections(userId));

        return result;

    }

    public void createNewCollection(String collectionTitle, String userId) {
        this.doCheckUserAuth(userId);
        collectionsDAO.createNewCollection(collectionTitle, userId);

    }

    public void deleteCollectionById(Long collectionId, String userId) {
        this.doCheckUserAuth(userId);

        if (collectionsDAO.deleteCollection(collectionId, userId) != 1) {
            throw new InvalidArgumentException("Collection not found");
        }
    }

    @Transactional(rollbackFor = {Exception.class}, isolation = Isolation.REPEATABLE_READ)
    public List<NodeDetails> getNodeDetailsInCollection(Long collectionId, String userId) {
        this.doCheckUserAuth(userId);

        Optional<NodeCollection> maybeNc
                = collectionsDAO.getNodeCollectionById(collectionId);

        NodeCollection nc = maybeNc.orElseThrow(
                () -> {
                    return new InvalidArgumentException("Collection not found");
                });

        if (!nc.getOwnerId().equals(userId)) {
            throw new PermissionDeniedException("User doesn't own collection");
        }

        return collectionsDAO.getNodeDetailsInCollection(nc.getId());

    }

    @Transactional(rollbackFor = {Exception.class}, isolation = Isolation.REPEATABLE_READ)
    public void addNodeToCollection(String vosPath, Long collectionId,
            String userId, List<String> userGroups) {
        this.doCheckUserAuth(userId);

        this.checkCollectionForOperations(collectionId, userId);

        Long nodeId = this.checkNodeForOperations(vosPath, userId, userGroups);

        collectionsDAO.addNodeToCollection(nodeId, collectionId);

    }

    @Transactional(rollbackFor = {Exception.class}, isolation = Isolation.REPEATABLE_READ)
    public void removeNodeFromCollection(String vosPath, Long collectionId,
            String userId, List<String> userGroups) {
        this.doCheckUserAuth(userId);

        this.checkCollectionForOperations(collectionId, userId);

        Long nodeId = checkNodeForOperations(vosPath, userId, userGroups);

        if (collectionsDAO
                .removeNodeFromCollection(nodeId, collectionId) != 1) {
            throw new InvalidArgumentException("Node is not in collection");
        }

    }

    private void checkCollectionForOperations(Long collectionId, String userId) {
        Optional<NodeCollection> maybeNc
                = collectionsDAO.getNodeCollectionById(collectionId);

        NodeCollection nc = maybeNc
                .orElseThrow(() -> {
                    return new InvalidArgumentException("collection doesn't exist");
                });

        if (!nc.getOwnerId().equals(userId)) {
            throw new PermissionDeniedException("User doesn't own collection");
        }
    }

    private Long checkNodeForOperations(String vosPath,
            String userId, List<String> userGroups) {
        Optional<ShortNodeDescriptor> maybeSnd
                = nodeDAO.getShortNodeDescriptor(vosPath, userId, userGroups);

        ShortNodeDescriptor snd = maybeSnd
                .orElseThrow(() -> {
                    return new NodeNotFoundException(vosPath);
                });

        // check if node is owned by the user and not busy
        if (!snd.getCreatorId().equals(userId)) {
            throw new PermissionDeniedException(
                    "User doesn't own node");
        }

        if (snd.isBusy()) {
            throw new NodeBusyException(vosPath);
        }

        return snd.getNodeId();
    }

    private void doCheckUserAuth(String userId) {
        if (!isUserAuthenticated(userId)) {
            throw new PermissionDeniedException("Authentication required");
        }
    }

    private boolean isUserAuthenticated(String userId) {
        return userId != null
                && !userId.equals("anonymous");
    }
}
