/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace;

import it.inaf.ia2.aa.data.User;
import it.inaf.oats.vospace.datamodel.NodeUtils;
import it.inaf.oats.vospace.exception.DuplicateNodeException;
import it.inaf.oats.vospace.exception.InternalFaultException;
import it.inaf.oats.vospace.exception.NodeBusyException;
import it.inaf.oats.vospace.exception.NodeNotFoundException;
import it.inaf.oats.vospace.exception.PermissionDeniedException;
import it.inaf.oats.vospace.persistence.NodeDAO.ShortNodeDescriptor;
import java.util.Optional;
import net.ivoa.xml.vospace.v2.Transfer;
import org.springframework.dao.CannotSerializeTransactionException;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.EnableTransactionManagement;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Transactional;

@Service
@EnableTransactionManagement
public class MoveService extends AbstractNodeService {
    
    /**
     * Perform modeNode operation. User is passed as parameter because this method
     * is run in a separate thread and original HttpServletRequest is not available
     * anymore ("No thread-bound request found" would happen).
     */
    @Transactional(rollbackFor = { Exception.class }, isolation = Isolation.REPEATABLE_READ)
    public void processMoveJob(Transfer transfer, User user) {

        // Get Source Vos Path
        String sourcePath = URIUtils.returnVosPathFromNodeURI(transfer.getTarget(), authority);

        // Get Destination Vos Path (it's in transfer direction)
        String destinationPath = URIUtils.returnVosPathFromNodeURI(transfer.getDirection(), authority);

        // Generic common validation for move process job paths
        this.validatePath(sourcePath);
        this.validatePath(destinationPath);

        if (sourcePath.equals(destinationPath)) {
            throw new IllegalArgumentException("Cannot move node to itself");
        }
        
        // Check if destination is subpath of source
        // Linux-like: "cannot move to a subdirectory of itself" 
        if(destinationPath.startsWith(sourcePath+"/")) {
            throw new IllegalArgumentException("Cannot move node to a subdirectory of its own path");
        }

        // Check if destination equals parent path of source
        if(NodeUtils.getParentPath(sourcePath).equals(destinationPath)){
            return;
        }        

        try {
            // Get source node
            Optional<Long> sourceIdOpt = nodeDao.getNodeId(sourcePath);
            long sourceId = sourceIdOpt.orElseThrow(() -> new NodeNotFoundException(sourcePath));

            if (nodeDao.isBranchBusy(sourceId)) {
                throw new NodeBusyException(sourcePath);
            }

            // TODO create immutable node exception flavor
            if (nodeDao.isBranchImmutable(sourceId)) {
                throw new InternalFaultException("Source branch contains immutable nodes");
            }

            if (!nodeDao.isBranchWritable(sourceId, user.getName(), user.getGroups())) {
                throw PermissionDeniedException.forPath(sourcePath);
            }           

            Optional<ShortNodeDescriptor> destShortNodeDescriptor 
                    = nodeDao.getShortNodeDescriptor(destinationPath, user.getName(), user.getGroups());

            String  destinationNodeLtreePath = null;
            if (destShortNodeDescriptor.isPresent()) {
                // When the destination is an existing ContainerNode, the source SHALL be placed under it (i.e., within the container)
                ShortNodeDescriptor snd = destShortNodeDescriptor.get();
                
                if(snd.isBusy()) throw new NodeBusyException(destinationPath);
                if(snd.isPermissionDenied()) throw PermissionDeniedException.forPath(destinationPath);                                                
                if(!snd.isWritable()) throw new InternalFaultException("Destination is not writable: "+ destinationPath);
                if(!snd.isContainer()) throw new InternalFaultException("Existing destination is not a container: " + destinationPath);
                if(snd.isImmutable()) throw new InternalFaultException("Destination is immutable: " + destinationPath);
                
                destinationNodeLtreePath = snd.getDestinationNodeLtreePath();
                
            } else {
                // Compare source and destination paths parents and see if it's just a rename        
                if (NodeUtils.getParentPath(sourcePath)
                        .equals(NodeUtils.getParentPath(destinationPath))) {

                    nodeDao.renameNode(sourceId, NodeUtils.getLastPathElement(destinationPath));
                } else {
                    throw new UnsupportedOperationException("Creation of destination upon move not supported");
                    // TODO (if we want this): modify the return type of createDestination to obtain an ltree path
                    // parentDestLtree = this.createDestination(NodeUtils.getParentPath(destinationPath), user);
                }
            }

            if (destinationNodeLtreePath != null) {                
                nodeDao.moveNodeBranch(sourceId, destinationNodeLtreePath);
            }
            
        } catch (CannotSerializeTransactionException ex) {
            // Concurrent transactions attempted to modify this set of nodes            
            throw new NodeBusyException(sourcePath);
        } catch (DuplicateKeyException ex) {
            throw new DuplicateNodeException(destinationPath);
        }
    }  

}
