/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace;

import it.inaf.oats.vospace.datamodel.NodeUtils;
import it.inaf.oats.vospace.exception.InvalidArgumentException;
import it.inaf.oats.vospace.exception.InvalidURIException;
import javax.servlet.http.HttpServletRequest;
import net.ivoa.xml.vospace.v2.LinkNode;
import net.ivoa.xml.vospace.v2.Node;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import static org.springframework.web.servlet.function.RequestPredicates.path;

public abstract class BaseNodeController {

    @Autowired
    private HttpServletRequest servletRequest;

    @Value("${vospace-authority}")
    protected String authority;

    protected String getPath() {
        String requestURL = servletRequest.getRequestURL().toString();
        try {
            return NodeUtils.getPathFromRequestURLString(requestURL);
        } catch (IllegalArgumentException ex) {
            throw new InvalidURIException(ex);
        }
    }

    protected String getParentPath(String path) {
        return NodeUtils.getParentPath(path);
    }

    protected void validateAndCheckPayloadURIConsistence(Node node) {
        // Get Node path (and validates it too)
        String decodedURIPathFromNode = URIUtils.returnVosPathFromNodeURI(node.getUri(), this.authority);

        // Check if payload URI is consistent with http request
        String requestPath = this.getPath();
        if (!decodedURIPathFromNode.equals(this.getPath())) {
            throw new InvalidURIException(decodedURIPathFromNode, requestPath);
        }

    }

    protected void validateLinkNode(LinkNode linkNode) {
        String target = linkNode.getTarget();
        // I validate it here to add context easily
        if (target == null) {
            throw new InvalidArgumentException("LinkNode in payload has no target element specified");
        }

        if (URIUtils.isURIInternal(target)) {
            URIUtils.returnVosPathFromNodeURI(linkNode.getTarget(), authority);
        } else {
            // TODO: Let's discuss if we need to combine this validation with
            // protocol endpoints management (URIService, ProtocolType)
            // Let's start with http and https only for now
            if (!(target.toLowerCase().startsWith("http://")
                    || target.toLowerCase().startsWith("https://"))) {
                throw new InvalidArgumentException("LinkNode target malformed or unsupported protocol: " + target);
            }

        }
    }
}
