/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace.persistence;

import it.inaf.oats.vospace.JobService;
import it.inaf.oats.vospace.datamodel.Views;
import java.util.List;
import javax.sql.DataSource;
import net.ivoa.xml.uws.v1.ExecutionPhase;
import net.ivoa.xml.uws.v1.JobSummary;
import net.ivoa.xml.uws.v1.ShortJobDescription;
import net.ivoa.xml.vospace.v2.Transfer;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.context.junit.jupiter.SpringExtension;
import java.util.Optional;
import java.time.LocalDateTime;
import java.time.Month;
import net.ivoa.xml.uws.v1.ErrorSummary;
import net.ivoa.xml.uws.v1.Jobs;
import it.inaf.oats.vospace.exception.ErrorSummaryFactory;
import it.inaf.oats.vospace.exception.PermissionDeniedException;
import net.ivoa.xml.uws.v1.ResultReference;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;

@ExtendWith(SpringExtension.class)
@ContextConfiguration(classes = {DataSourceConfig.class})
@TestPropertySource(locations = "classpath:test.properties")
public class JobDAOTest {

    @Autowired
    private DataSource dataSource;
    private JobDAO dao;

    @BeforeEach
    public void init() {
        dao = new JobDAO(dataSource);
    }

    private JobSummary getJob() {
        Transfer transfer = new Transfer();
        transfer.setDirection("pushToVoSpace");
        transfer.setTarget("vos://example.com!vospace/mynode");

        JobSummary job = new JobSummary();
        job.setJobId("123");
        job.setOwnerId("owner");
        job.setPhase(ExecutionPhase.PENDING);
        JobSummary.JobInfo jobInfo = new JobSummary.JobInfo();
        jobInfo.getAny().add(transfer);
        job.setJobInfo(jobInfo);

        return job;
    }

    // don't want to override equals method just for testing without a use case
    private boolean areEqual(ErrorSummary a, ErrorSummary b) {     
        return (a.getMessage().equals(b.getMessage())
                && a.getType().equals(b.getType())
                && a.isHasDetail() == b.isHasDetail()
                && a.getDetailMessage().equals(b.getDetailMessage()));
    }

    @Test
    public void testJob() {

        JobSummary job = getJob();

        dao.createJob(job, null);

        assertTrue(dao.getJob("123").isPresent());
        assertEquals(ExecutionPhase.PENDING, dao.getJob("123").get().getPhase());

        // uses the job retrieved from DAO to perform the update (reproduced a bug in job update)
        job = dao.getJob("123").get();
        assertNotNull(job.getCreationTime());
        assertNull(job.getStartTime());

        job.setPhase(ExecutionPhase.EXECUTING);
        dao.updateJob(job, null);

        job = dao.getJob("123").get();
        assertEquals(ExecutionPhase.EXECUTING, job.getPhase());
        assertNotNull(job.getStartTime());
        assertNull(job.getEndTime());

        assertNull(dao.getTransferDetails(job.getJobId()));

        Transfer negotiatedTransfer = new Transfer();
        job.setPhase(ExecutionPhase.COMPLETED);
        dao.updateJob(job, negotiatedTransfer);

        job = dao.getJob("123").get();
        assertEquals(ExecutionPhase.COMPLETED, job.getPhase());
        assertNotNull(job.getStartTime());
        assertNotNull(job.getEndTime());
        assertNotNull(dao.getTransferDetails(job.getJobId()));
    }
    
    /**
     * Jobs created by the /synctrans endpoint contains results list at creation time.
     */
    @Test
    public void testCreateJobWithResults() {
        JobSummary job = getJob();
        job.setPhase(ExecutionPhase.COMPLETED);

        ResultReference result = new ResultReference();
        result.setId("transferDetails");
        result.setHref("http://ia2.inaf.it");
        job.getResults().add(result);

        Transfer negotiatedTransfer = new Transfer();

        dao.createJob(job, negotiatedTransfer);

        // Retrieve it back        
        Optional<JobSummary> retrievedJobOpt = dao.getJob(job.getJobId());
        assertTrue(retrievedJobOpt.isPresent());

        JobSummary retrievedJob = retrievedJobOpt.get();
        assertEquals(1, retrievedJob.getResults().size());
        assertNotNull(retrievedJob.getStartTime());
        assertNotNull(retrievedJob.getEndTime());

        assertNotNull(dao.getTransferDetails(retrievedJob.getJobId()));
    }

    @Test
    public void testCreateJobWithError() {
        JobSummary job = getJob();

        job.setPhase(ExecutionPhase.ERROR);

        // Generate it from exception        
        ErrorSummary errorSummary
                = ErrorSummaryFactory.newErrorSummary(
                        PermissionDeniedException.forPath("/pippo1/pippo2"));
        
        // Check if properly generated
        assertTrue(errorSummary.isHasDetail());
        assertEquals("PermissionDenied Path: /pippo1/pippo2", errorSummary.getDetailMessage());

        job.setErrorSummary(errorSummary);

        dao.createJob(job, null);

        // Retrieve it back        
        Optional<JobSummary> retrievedJobOpt = dao.getJob(job.getJobId());
        assertTrue(retrievedJobOpt.isPresent());

        JobSummary retrievedJob = retrievedJobOpt.get();
        assertEquals(ExecutionPhase.ERROR, retrievedJob.getPhase());
        assertTrue(areEqual(job.getErrorSummary(), retrievedJob.getErrorSummary()));
        assertNotNull(retrievedJob.getStartTime());
        assertNotNull(retrievedJob.getEndTime());
    }

    @Test
    public void testUpdateJobWithError() {
        JobSummary job = getJob();

        dao.createJob(job, null);

        job.setPhase(ExecutionPhase.ERROR);
        // Generate it from exception        
        ErrorSummary errorSummary
                = ErrorSummaryFactory.newErrorSummary(
                        PermissionDeniedException.forPath("/pippo1/pippo2"));
        
        // Check if properly generated        
        assertTrue(errorSummary.isHasDetail());
        assertEquals("PermissionDenied Path: /pippo1/pippo2", errorSummary.getDetailMessage());
                
        job.setErrorSummary(errorSummary);

        dao.updateJob(job, null);

        // Retrieve it back        
        Optional<JobSummary> retrievedJobOpt = dao.getJob(job.getJobId());
        assertTrue(retrievedJobOpt.isPresent());

        JobSummary retrievedJob = retrievedJobOpt.get();
        assertEquals(ExecutionPhase.ERROR, retrievedJob.getPhase());
        assertTrue(areEqual(job.getErrorSummary(), retrievedJob.getErrorSummary()));
        assertNotNull(retrievedJob.getEndTime());
    }

    @Test
    public void testJobsList() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList = List.of();
        List<JobService.JobDirection> directionList = List.of();
        List<String> viewList = List.of();
        Optional<LocalDateTime> after = Optional.ofNullable(null);
        Optional<Integer> last = Optional.ofNullable(null);

        Jobs jobs = dao.getJobs(user, phaseList, directionList, viewList, after, last);

        assertTrue(jobs != null);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertTrue(sjdList != null);
        assertTrue(!sjdList.isEmpty());
        assertTrue(
                sjdList.stream().noneMatch(
                        (i) -> {
                            return i.getPhase().equals(ExecutionPhase.ARCHIVED);
                        }
                )
        );
        assertEquals(3, sjdList.size());
    }

    @Test
    public void testJobsListLimit() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList = List.of();
        List<JobService.JobDirection> directionList = List.of();
        List<String> viewList = List.of();
        Optional<LocalDateTime> after = Optional.ofNullable(null);
        Optional<Integer> last = Optional.of(2);

        Jobs jobs = dao.getJobs(user, phaseList, directionList, viewList, after, last);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertEquals(2, sjdList.size());

    }

    @Test
    public void testJobsPhase() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList
                = List.of(ExecutionPhase.PENDING, ExecutionPhase.EXECUTING);
        List<JobService.JobDirection> directionList = List.of();
        List<String> viewList = List.of();
        Optional<LocalDateTime> after = Optional.ofNullable(null);
        Optional<Integer> last = Optional.ofNullable(null);

        Jobs jobs = dao.getJobs(user, phaseList, directionList, viewList, after, last);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertEquals(sjdList.size(), 2);
        assertEquals("pippo5", sjdList.get(0).getId());
        assertEquals("pippo2", sjdList.get(1).getId());
    }

    @Test
    public void testJobsDirection() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList = List.of();
        List<JobService.JobDirection> directionList
                = List.of(JobService.JobDirection.pullFromVoSpace,
                        JobService.JobDirection.pullToVoSpace);
        List<String> viewList = List.of();

        Optional<LocalDateTime> after = Optional.ofNullable(null);
        Optional<Integer> last = Optional.ofNullable(null);
        Jobs jobs = dao.getJobs(user, phaseList, directionList, viewList, after, last);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertEquals(2, sjdList.size());
        assertEquals("pippo3", sjdList.get(0).getId());
        assertEquals("pippo2", sjdList.get(1).getId());
    }

    @Test
    public void testJobsAfter() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList = List.of();
        List<JobService.JobDirection> directionList = List.of();
        List<String> viewList = List.of();

        LocalDateTime ldt
                = LocalDateTime.of(2013, Month.FEBRUARY, 7, 18, 15);
        Optional<LocalDateTime> after = Optional.of(ldt);

        Optional<Integer> last = Optional.ofNullable(null);
        Jobs jobs = dao.getJobs(user, phaseList, directionList, viewList, after, last);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertEquals(2, sjdList.size());
        assertEquals("pippo5", sjdList.get(0).getId());
        assertEquals("pippo3", sjdList.get(1).getId());
    }

    @Test
    public void testJobsAllchecks() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList = List.of(ExecutionPhase.QUEUED,
                ExecutionPhase.PENDING);
        List<JobService.JobDirection> directionList
                = List.of(JobService.JobDirection.pullFromVoSpace,
                        JobService.JobDirection.pullToVoSpace);
        List<String> viewList = List.of(Views.TAR_VIEW_URI, Views.ZIP_VIEW_URI);

        LocalDateTime ldt
                = LocalDateTime.of(2013, Month.FEBRUARY, 7, 18, 15);
        Optional<LocalDateTime> after = Optional.of(ldt);

        Optional<Integer> last = Optional.of(2);
        Jobs jobs = dao.getJobs(user, phaseList, directionList, viewList, after, last);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertEquals(1, sjdList.size());
        assertEquals("pippo3", sjdList.get(0).getId());
    }
}
