/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace;

import it.inaf.ia2.aa.data.User;
import it.inaf.oats.vospace.exception.NodeBusyException;
import it.inaf.oats.vospace.exception.NodeNotFoundException;
import it.inaf.oats.vospace.exception.PermissionDeniedException;
import it.inaf.oats.vospace.persistence.DataSourceConfigSingleton;
import it.inaf.oats.vospace.persistence.NodeDAO;
import java.util.List;
import java.util.Optional;
import net.ivoa.xml.vospace.v2.Transfer;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.MethodOrderer.OrderAnnotation;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.context.ContextConfiguration;

@SpringBootTest
@AutoConfigureMockMvc
@ContextConfiguration(classes = {DataSourceConfigSingleton.class})
@TestPropertySource(locations = "classpath:test.properties", properties = {"vospace-authority=example.com!vospace", "file-service-url=http://file-service"})
@TestMethodOrder(OrderAnnotation.class)
@DirtiesContext(classMode = DirtiesContext.ClassMode.AFTER_EACH_TEST_METHOD)
public class MoveServiceTest {

    @Value("${vospace-authority}")
    private String authority;

    @Autowired
    private MoveService moveService;

    @Autowired
    private NodeDAO nodeDao;

    @Test
    @Order(1)
    public void moveRootTest() {

        assertThrows(IllegalArgumentException.class, () -> {
            moveService.processMoveJob(getTransfer("/", "/pippo"), getAnonymousUser());
        }
        );

        assertThrows(IllegalArgumentException.class, () -> {
            moveService.processMoveJob(getTransfer("/pippo", "/"), getAnonymousUser());
        }
        );

    }

    @Test
    @Order(2)
    public void testMoveToSubnodeOfItself() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user3");

        assertThrows(IllegalArgumentException.class, () -> {
            moveService.processMoveJob(getTransfer("/test3/m1", "/test3/m1/m2"), user);
        }
        );
    }

    @Test
    @Order(3)
    public void testNonExistingSourceNode() {
        assertThrows(NodeNotFoundException.class, () -> {
            moveService.processMoveJob(getTransfer("/pippo", "/test2"), getAnonymousUser());
        }
        );
    }

    @Test
    @Order(4)
    public void testMoveDeniedOnBusySource() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user3");

        assertThrows(NodeBusyException.class, () -> {
            moveService.processMoveJob(getTransfer("/test3/mbusy", "/test3/m1"), user);
        }
        );
    }

    @Test
    @Order(5)
    public void testPermissionDeniedOnSource() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user1");

        assertThrows(PermissionDeniedException.class, () -> {
            moveService.processMoveJob(getTransfer("/test3/m1", "/test4"), user);
        }
        );
    }

    @Test
    @Order(5)
    public void testDontMoveIfStickySource() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user3");

        assertThrows(PermissionDeniedException.class, () -> {
            moveService.processMoveJob(getTransfer("/test3/mstick", "/test4"), user);
        }
        );
    }

    @Test
    @Order(6)
    public void testPermissionDeniedOnExistingDestination() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user1");
        when(user.getGroups()).thenReturn(List.of("group1"));

        assertThrows(PermissionDeniedException.class, () -> {
            moveService.processMoveJob(getTransfer("/test3/group1", "/test3/m1/m2"), user);
        }
        );
    }

    @Test
    @Order(7)
    public void testDestinationExistsAndIsBusy() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user3");

        assertThrows(NodeBusyException.class, () -> {
            moveService.processMoveJob(getTransfer("/test3/m1", "/test3/mbusy"), user);
        }
        );
    }

    @Test
    @Order(8)
    public void testRenameNode() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user3");

        Optional<Long> sourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(sourceId.isPresent());
        Optional<Long> childId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(childId.isPresent());
        // Rename
        moveService.processMoveJob(getTransfer("/test3/m1", "/test3/m1ren"), user);

        Optional<Long> checkSourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(checkSourceId.isEmpty());

        Optional<Long> newSourceId = nodeDao.getNodeId("/test3/m1ren");
        assertTrue(newSourceId.isPresent());
        assertEquals(sourceId.get(), newSourceId.get());

        Optional<Long> newChildId = nodeDao.getNodeId("/test3/m1ren/m2");
        assertTrue(newChildId.isPresent());
        assertEquals(childId.get(), newChildId.get());

    }

    @Test
    @Order(9)
    public void testMoveToExistingDestination() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user3");

        // Preliminary checks for assumptions
        Optional<Long> sourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(sourceId.isPresent());
        Optional<Long> childId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(childId.isPresent());

        Optional<Long> destParentId = nodeDao.getNodeId("/test4");
        assertTrue(destParentId.isPresent());

        Optional<Long> destId = nodeDao.getNodeId("/test4");
        assertTrue(destId.isPresent());

        // move
        moveService.processMoveJob(getTransfer("/test3/m1", "/test4"), user);

        // source has been moved
        Optional<Long> oldSourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(oldSourceId.isEmpty());
        Optional<Long> oldChildId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(oldChildId.isEmpty());

        Optional<Long> newSourceId = nodeDao.getNodeId("/test4/m1");
        assertTrue(newSourceId.isPresent());
        assertEquals(sourceId.get(), newSourceId.get());

        Optional<Long> newChildId = nodeDao.getNodeId("/test4/m1/m2");
        assertTrue(newChildId.isPresent());
        assertEquals(childId.get(), newChildId.get());

    }

    private Transfer getTransfer(String vosTarget, String vosDestination) {
        Transfer transfer = new Transfer();
        transfer.setTarget("vos://" + this.authority + vosTarget);
        transfer.setDirection("vos://" + this.authority + vosDestination);
        return transfer;
    }

    private User getAnonymousUser() {
        return new User().setUserId("anonymous");
    }
}
