/*
 * This file is part of vospace-datamodel
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace;

import it.inaf.oats.vospace.exception.InvalidURIException;
import java.net.URISyntaxException;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;
import org.junit.jupiter.api.Test;

public class URIUtilsTest {
    
    private final String authority = "example.com!vospace";
    
    @Test
    public void testReturnVosPathFromNodeURI() {
        
        String test1 = "vos://example.com!wrong/";
        
        assertThrows(InvalidURIException.class, () -> {
            URIUtils.returnVosPathFromNodeURI(test1, authority);
        });
        
        String test2 = "wrong://example.com!wrong/";
        
        assertThrows(InvalidURIException.class, () -> {
            URIUtils.returnVosPathFromNodeURI(test2, authority);
        });
        
        String test3 = "vos://example.com!vospace";
        
        assertThrows(InvalidURIException.class, () -> {
            URIUtils.returnVosPathFromNodeURI(test3, authority);
        });
        
        String test4 = "vos://example.com!vospace/n1/n2/n3/n%2F4";
        
        assertThrows(InvalidURIException.class, () -> {
            URIUtils.returnVosPathFromNodeURI(test4, authority);
        });
        
        String test5 = "vos://example.com!vospace/n1/n2/n3//n4";
        
        assertThrows(InvalidURIException.class, () -> {
            URIUtils.returnVosPathFromNodeURI(test5, authority);
        });
        
        String test6 = "vos://example.com!vospace/n1/n2/n3/n4/";
        
        assertThrows(InvalidURIException.class, () -> {
            URIUtils.returnVosPathFromNodeURI(test6, authority);
        });
        
        String test7 = "vos://example.com!vospace/n1/n2/n*3/n4/";
        
        assertThrows(InvalidURIException.class, () -> {
            URIUtils.returnVosPathFromNodeURI(test7, authority);
        });                
        
        String test8 = "vos://example.com!vospace/n1/n2/n3/n4";
        
        assertEquals("/n1/n2/n3/n4", 
                URIUtils.returnVosPathFromNodeURI(test8, authority));
        
        String test9 = "vos://example.com!vospace/";
        
        assertEquals("/", 
                URIUtils.returnVosPathFromNodeURI(test9, authority));        
        
    }
    
    @Test
    public void testReturnURIFromVosPath() throws URISyntaxException
    {
        String test1 = URIUtils.returnURIFromVosPath("/", authority);        
        assertEquals("vos://"+authority+"/", test1);
        
        String test2 = URIUtils.returnURIFromVosPath("/test1/test2", authority);        
        assertEquals("vos://"+authority+"/test1/test2", test2);
        
        String test3 = URIUtils.returnURIFromVosPath("/test1/te# !?st2", authority);
        assertEquals("vos://"+authority+"/test1/te%23%20!%3Fst2", test3);       
    }

    
}
