/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.inaf.ia2.aa.data.User;
import it.inaf.oats.vospace.FileServiceClient.ArchiveRequest;
import it.inaf.oats.vospace.datamodel.Views;
import it.inaf.oats.vospace.exception.InvalidArgumentException;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URI;
import java.util.Arrays;
import javax.servlet.http.HttpServletRequest;
import net.ivoa.xml.vospace.v2.Param;
import net.ivoa.xml.vospace.v2.Transfer;
import net.ivoa.xml.vospace.v2.View;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.fail;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import org.mockito.junit.jupiter.MockitoExtension;
import org.mockito.junit.jupiter.MockitoSettings;
import org.mockito.quality.Strictness;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.client.ClientHttpRequest;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.test.util.ReflectionTestUtils;
import org.springframework.web.client.RequestCallback;
import org.springframework.web.client.ResponseExtractor;
import org.springframework.web.client.RestTemplate;

@ExtendWith(MockitoExtension.class)
@MockitoSettings(strictness = Strictness.LENIENT)
public class FileServiceClientTest {

    private static final ObjectMapper MAPPER = new ObjectMapper();

    @Mock
    private RestTemplate restTemplate;

    @Mock
    private HttpServletRequest request;

    @InjectMocks
    private FileServiceClient fileServiceClient;

    @BeforeEach
    public void setUp() {
        ReflectionTestUtils.setField(fileServiceClient, "authority", "example.com!vospace");
        ReflectionTestUtils.setField(fileServiceClient, "fileServiceUrl", "http://file-service");
    }

    @Test
    public void testTarArchiveJob() {
        testStartArchiveJob(Views.TAR_VIEW_URI);
    }

    @Test
    public void testZipArchiveJob() {
        testStartArchiveJob(Views.ZIP_VIEW_URI);
    }

    @Test
    public void testInvalidArchiveJob() {
        try {
            testStartArchiveJob("foo");
            fail("IllegalArgumentException was expected");
        } catch (IllegalArgumentException ex) {
        }
    }

    @Test
    public void testArchiveNoInclude() {

        Transfer transfer = new Transfer();
        transfer.setDirection("pullFromVoSpace");
        transfer.setTarget(Arrays.asList("vos://example.com!vospace/mydir"));
        View view = new View();
        view.setUri(Views.ZIP_VIEW_URI);
        transfer.setView(view);

        ArchiveRequest archiveRequest = testStartArchiveJob(transfer);

        assertEquals(1, archiveRequest.getPaths().size());
        assertEquals("/mydir", archiveRequest.getPaths().get(0));
    }

    @Test
    public void testInvalidViewParam() {

        Transfer transfer = new Transfer();
        transfer.setDirection("pullFromVoSpace");
        transfer.setTarget(Arrays.asList("vos://example.com!vospace/parent_dir"));
        View view = new View();
        view.setUri(Views.TAR_VIEW_URI);
        transfer.setView(view);

        Param param1 = new Param();
        param1.setUri("invalid");
        param1.setValue("file1");
        view.getParam().add(param1);

        assertThrows(InvalidArgumentException.class, () -> testStartArchiveJob(transfer));
    }

    @Test
    public void testInvalidViewParamPath() {

        Transfer transfer = new Transfer();
        transfer.setDirection("pullFromVoSpace");
        transfer.setTarget(Arrays.asList("vos://example.com!vospace/parent_dir"));
        View view = new View();
        view.setUri(Views.TAR_VIEW_URI);
        transfer.setView(view);

        Param param1 = new Param();
        param1.setUri(Views.TAR_VIEW_URI + "/include");
        param1.setValue("../file1");
        view.getParam().add(param1);

        assertThrows(InvalidArgumentException.class, () -> testStartArchiveJob(transfer));
    }

    private void testStartArchiveJob(String viewUri) {

        Transfer transfer = new Transfer();
        transfer.setDirection("pullFromVoSpace");
        transfer.setTarget(Arrays.asList("vos://example.com!vospace/parent_dir"));
        View view = new View();
        view.setUri(viewUri);
        transfer.setView(view);

        Param param1 = new Param();
        param1.setUri(viewUri + "/include");
        param1.setValue("file1");
        view.getParam().add(param1);

        Param param2 = new Param();
        param2.setUri(viewUri + "/include");
        param2.setValue("file2");
        view.getParam().add(param2);

        ArchiveRequest archiveRequest = testStartArchiveJob(transfer);

        assertEquals(2, archiveRequest.getPaths().size());
        assertEquals("/parent_dir/file1", archiveRequest.getPaths().get(0));
        assertEquals("/parent_dir/file2", archiveRequest.getPaths().get(1));
    }

    private ArchiveRequest testStartArchiveJob(Transfer transfer) {

        User user = mock(User.class);
        when(user.getAccessToken()).thenReturn("<token>");
        when(request.getUserPrincipal()).thenReturn(user);

        ByteArrayOutputStream baos = new ByteArrayOutputStream();

        doAnswer(invocation -> {
            RequestCallback requestCallback = invocation.getArgument(2);
            ClientHttpRequest mockedRequest = mock(ClientHttpRequest.class);
            HttpHeaders mockedRequestHeaders = mock(HttpHeaders.class);
            when(mockedRequest.getBody()).thenReturn(baos);
            when(mockedRequest.getHeaders()).thenReturn(mockedRequestHeaders);
            requestCallback.doWithRequest(mockedRequest);

            ResponseExtractor responseExtractor = invocation.getArgument(3);
            ClientHttpResponse mockedResponse = mock(ClientHttpResponse.class);
            HttpHeaders mockedResponseHeaders = mock(HttpHeaders.class);
            when(mockedResponseHeaders.getLocation()).thenReturn(new URI("http://file-service/archive/result"));
            when(mockedResponse.getHeaders()).thenReturn(mockedResponseHeaders);
            responseExtractor.extractData(mockedResponse);
            return mockedResponseHeaders.getLocation().toString();
        }).when(restTemplate).execute(eq("http://file-service/archive"), eq(HttpMethod.POST),
                any(RequestCallback.class), any(ResponseExtractor.class), any(Object[].class));

        String redirect = fileServiceClient.startArchiveJob(transfer, "job123");

        assertEquals("http://file-service/archive/result", redirect);

        try {
            return MAPPER.readValue(baos.toByteArray(), ArchiveRequest.class);
        } catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }
}
