/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace;

import it.inaf.ia2.aa.data.User;
import it.inaf.oats.vospace.datamodel.NodeUtils;
import it.inaf.oats.vospace.exception.InternalFaultException;
import it.inaf.oats.vospace.exception.InvalidArgumentException;
import it.inaf.oats.vospace.exception.NodeBusyException;
import it.inaf.oats.vospace.exception.NodeNotFoundException;
import it.inaf.oats.vospace.exception.PermissionDeniedException;
import it.inaf.oats.vospace.persistence.NodeDAO;
import it.inaf.oats.vospace.persistence.NodeDAO.ShortNodeDescriptor;
import java.util.Optional;
import javax.servlet.http.HttpServletRequest;
import net.ivoa.xml.vospace.v2.Transfer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.dao.CannotSerializeTransactionException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.EnableTransactionManagement;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

@Service
@EnableTransactionManagement
public class NodeBranchService {

    @Autowired
    private NodeDAO nodeDao;

    @Transactional(rollbackFor = {Exception.class}, isolation = Isolation.REPEATABLE_READ,
            propagation = Propagation.REQUIRED)
    public void checkBranchForReadAndLock(String sourcePath, String jobId, User user) {

        try {
            // Get source node
            Optional<Long> sourceIdOpt = nodeDao.getNodeId(sourcePath);
            long sourceId = sourceIdOpt.orElseThrow(() -> new NodeNotFoundException(sourcePath));

            if (nodeDao.isBranchBusy(sourceId)) {
                throw new NodeBusyException(sourcePath);
            }

            if (!nodeDao.isBranchReadable(sourceId, user.getName(), user.getGroups())) {                
                throw new PermissionDeniedException(sourcePath);
            }
            
            this.lockBranch(sourceId, jobId);

        } catch (CannotSerializeTransactionException ex) {
            // Concurrent transactions attempted to modify this set of nodes            
            throw new NodeBusyException(sourcePath);
        }
    } 
    
    private void lockBranch(Long sourceRootId, String jobId) {

        nodeDao.setBranchJobId(sourceRootId, jobId);

    }
    
    public void unlockNodes(String jobId) {
        nodeDao.releaseBusyNodesByJobId(jobId);
    }

}
