/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace;

import it.inaf.oats.vospace.datamodel.Views;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import net.ivoa.xml.uws.v1.JobSummary;
import net.ivoa.xml.vospace.v2.Param;
import net.ivoa.xml.vospace.v2.Transfer;
import net.ivoa.xml.vospace.v2.View;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.fail;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import static org.mockito.ArgumentMatchers.anyInt;
import static org.mockito.ArgumentMatchers.anyString;
import org.mockito.Mock;
import org.mockito.MockedConstruction;
import org.mockito.Mockito;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.when;
import org.mockito.junit.jupiter.MockitoExtension;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

@ExtendWith(MockitoExtension.class)
public class AsyncTransferServiceTest {

    private static final String JSON_JOB = "{\"jobId\":\"job_id\",\"runId\":null,\"ownerId\":null,\"phase\":null,\"quote\":null,\"creationTime\":null,\"startTime\":null,\"endTime\":null,\"executionDuration\":0,\"destruction\":null,\"parameters\":null,\"results\":[],\"errorSummary\":null,\"jobInfo\":{\"transfer\":{\"target\":\"vos://example.com!vospace/my-node\",\"direction\":\"pullToVoSpace\",\"view\":{\"param\":[{\"value\":\"file1.txt\",\"uri\":\"ivo://ia2.inaf.it/vospace/views#async-recall/include\"},{\"value\":\"file2.txt\",\"uri\":\"ivo://ia2.inaf.it/vospace/views#async-recall/include\"}],\"uri\":\"ivo://ia2.inaf.it/vospace/views#async-recall\",\"original\":true},\"protocols\":[],\"keepBytes\":false,\"version\":null,\"param\":[]}},\"version\":null}";

    @Mock
    private Jedis mockedJedis;

    @Test
    public void testRedisRpc() {

        doAnswer(invocation -> {
            String requestId = invocation.getArgument(1);
            List<String> result = new ArrayList<>();
            result.add(requestId);
            result.add(JSON_JOB);
            return result;
        }).when(mockedJedis).brpop(anyInt(), anyString());

        testWithMockedJedis(asyncTransferService -> {
            JobSummary result = asyncTransferService.startJob(getFakeJob());
            assertEquals("job_id", result.getJobId());
        });
    }

    @Test
    public void testRedisError() {

        when(mockedJedis.brpop(anyInt(), anyString())).thenReturn(List.of());

        testWithMockedJedis(asyncTransferService -> {
            try {
                asyncTransferService.startJob(getFakeJob());
                fail();
            } catch (IllegalStateException ex) {
            }
        });
    }

    private void testWithMockedJedis(Consumer<AsyncTransferService> testAction) {
        try (MockedConstruction<JedisPool> staticMock = Mockito.mockConstruction(JedisPool.class,
                (mockedJedisPool, context) -> {
                    when(mockedJedisPool.getResource()).thenReturn(mockedJedis);
                })) {
            AsyncTransferService asyncTransferService = new AsyncTransferService("localhost", 6379);
            testAction.accept(asyncTransferService);
        }
    }

    private JobSummary getFakeJob() {
        Transfer transfer = new Transfer();
        transfer.setDirection("pullToVoSpace");

        transfer.setTarget("vos://example.com!vospace/my-node");

        View view = new View();
        view.setUri(Views.ASYNC_RECALL_VIEW_URI);

        Param p1 = new Param();
        p1.setUri(Views.ASYNC_RECALL_VIEW_URI + "/include");
        p1.setValue("file1.txt");
        view.getParam().add(p1);

        Param p2 = new Param();
        p2.setUri(Views.ASYNC_RECALL_VIEW_URI + "/include");
        p2.setValue("file2.txt");
        view.getParam().add(p2);

        transfer.setView(view);

        JobSummary job = new JobSummary();
        job.setJobId("job_id");

        JobSummary.JobInfo jobInfo = new JobSummary.JobInfo();
        jobInfo.getAny().add(transfer);

        job.setJobInfo(jobInfo);

        return job;
    }
}
