/*
 * This file is part of vospace-file-service
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.transfer.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.inaf.ia2.transfer.auth.TokenPrincipal;
import it.inaf.ia2.transfer.persistence.JobDAO;
import it.inaf.ia2.transfer.service.FileCopyService;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;


@SpringBootTest
@AutoConfigureMockMvc(addFilters = false)
public class CopyControllerTest {

    @MockBean
    private JobDAO jobDao;

    @Autowired
    private MockMvc mockMvc;

    @MockBean
    private FileCopyService fileCopyService;
    
    private static String jobId;
    private static String sourceVosRootPath;
    private static String destVosRootPath;
    private static String filename;

    private static final ObjectMapper MAPPER = new ObjectMapper();

    @BeforeAll
    public static void setUp() {
        jobId = "pippo_copy";
        filename = "file.txt";
        sourceVosRootPath = "/source/" + filename;
        destVosRootPath = "/destination/" + filename;
    }

    @Test
    public void testCopyControllerNoJobMatching() throws Exception {

        CopyRequest copyRequest = this.getCopyRequest(jobId,
                sourceVosRootPath, destVosRootPath);

        when(jobDao.isJobExisting(eq(jobId))).thenReturn(false);
        
        mockMvc.perform(post("/copy")
                .principal(fakePrincipal("user_copy"))
                .contentType(MediaType.APPLICATION_JSON)
                .content(MAPPER.writeValueAsString(copyRequest)))
                .andDo(print())
                .andExpect(status().is4xxClientError());

    }
    
    @Test
    public void testCopyControllerAnonymousForbidden() throws Exception {

        CopyRequest copyRequest = this.getCopyRequest(jobId,
                sourceVosRootPath, destVosRootPath);

        when(jobDao.isJobExisting(eq(jobId))).thenReturn(true);

        mockMvc.perform(post("/copy")
                .principal(fakePrincipal("anonymous"))
                .contentType(MediaType.APPLICATION_JSON)
                .content(MAPPER.writeValueAsString(copyRequest)))
                .andDo(print())
                .andExpect(status().is4xxClientError());
    }

    private CopyRequest getCopyRequest(String jobId,
            String sourceRootVosPath, String destinationRootVosPath) {
        CopyRequest result = new CopyRequest();
        result.setJobId(jobId);
        result.setSourceRootVosPath(sourceRootVosPath);
        result.setDestinationRootVosPath(destinationRootVosPath);
        return result;
    }

    private TokenPrincipal fakePrincipal(String name) {
        TokenPrincipal principal = mock(TokenPrincipal.class);
        when(principal.getName()).thenReturn(name);
        return principal;
    }

}
