/*
 * This file is part of vospace-file-service
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.transfer.persistence;

import it.inaf.oats.vospace.exception.VoSpaceErrorSummarizableException;
import java.sql.Types;
import javax.sql.DataSource;
import net.ivoa.xml.uws.v1.ExecutionPhase;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Repository;

@Repository
public class JobDAO {

    private final JdbcTemplate jdbcTemplate;

    @Autowired
    public JobDAO(DataSource dataSource) {
        jdbcTemplate = new JdbcTemplate(dataSource);
    }

    public boolean isJobExisting(String jobId) {

        String sql = "SELECT * FROM job WHERE job_id = ?";

        boolean result = jdbcTemplate.query(sql,
                ps -> {
                    ps.setString(1, jobId);
                }, rs -> {
                    return rs.next();
                });

        return result;
    }

    public void updateJobPhase(ExecutionPhase phase, String jobId) {

        String sql = "UPDATE job SET phase = ? WHERE job_id = ?";

        jdbcTemplate.update(sql, ps -> {
            ps.setObject(1, phase, Types.OTHER);
            ps.setString(2, jobId);
        });
    }

    public ExecutionPhase getJobPhase(String jobId) {
        String sql = "SELECT phase FROM job WHERE job_id = ?";

        ExecutionPhase result = jdbcTemplate.query(sql,
                ps -> {
                    ps.setString(1, jobId);
                }, rs -> {
                    if (rs.next()) {
                        return ExecutionPhase.fromValue(rs.getString("phase"));
                    } else {
                        return null;
                    }
                });

        return result;
    }

    public void setJobError(String jobId, VoSpaceErrorSummarizableException jobError) {

        String sql = "UPDATE job SET phase = ?, error_message = ?, error_type = ?,\n"
                + "error_has_detail = ?, error_detail = ?, end_time = NOW()\n"
                + "WHERE job_id = ?";

        jdbcTemplate.update(sql, ps -> {
            int i = 0;
            ps.setObject(++i, ExecutionPhase.ERROR, Types.OTHER);
            ps.setString(++i, jobError.getMessage());
            ps.setObject(++i, jobError.getFault().getType().value(), Types.OTHER);
            ps.setBoolean(++i, jobError.getDetailMessage() != null);
            ps.setString(++i, jobError.getDetailMessage());
            ps.setString(++i, jobId);
        });
    }
}
