/*
 * This file is part of vospace-file-service
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.transfer.service;

import it.inaf.ia2.aa.ServletRapClient;
import it.inaf.ia2.rap.client.call.TokenExchangeRequest;
import it.inaf.ia2.transfer.auth.TokenPrincipal;
import it.inaf.ia2.transfer.persistence.FileDAO;
import it.inaf.ia2.transfer.persistence.JobDAO;
import it.inaf.ia2.transfer.persistence.model.FileInfo;
import it.inaf.oats.vospace.datamodel.NodeUtils;
import it.inaf.oats.vospace.exception.InternalFaultException;
import it.inaf.oats.vospace.exception.PermissionDeniedException;
import it.inaf.oats.vospace.exception.QuotaExceededException;
import it.inaf.oats.vospace.parent.exchange.ArchiveEntryDescriptor;
import it.inaf.oats.vospace.parent.persistence.LinkedServiceDAO;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.security.Principal;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import javax.annotation.PostConstruct;
import javax.servlet.http.HttpServletRequest;
import net.ivoa.xml.uws.v1.ExecutionPhase;
import org.kamranzafar.jtar.TarEntry;
import org.kamranzafar.jtar.TarOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.stereotype.Service;
import org.springframework.util.FileSystemUtils;
import org.springframework.util.unit.DataSize;
import org.springframework.web.client.RestTemplate;

@Service
public class ArchiveService {

    private static final Logger LOG = LoggerFactory.getLogger(ArchiveService.class);

    @Autowired
    private FileDAO fileDAO;

    @Autowired
    private LinkedServiceDAO linkedServiceDAO;

    @Autowired
    private JobDAO jobDAO;

    @Autowired
    private AuthorizationService authorizationService;

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private ServletRapClient rapClient;

    // Directory containing temporary files generated by jobs.
    @Value("${generated.dir}")
    private String generatedDirString;
    private File generatedDir;

    // Maximum size of the working directory for each registered user
    @Value("${generated.dir.max-size}")
    private DataSize generatedDirMaxSize;

    @PostConstruct
    public void init() {
        this.generatedDir = new File(generatedDirString);
        if (!this.generatedDir.exists()) {
            if (!this.generatedDir.mkdirs()) {
                throw new IllegalStateException("Unable to create directory " + this.generatedDir.getAbsolutePath());
            }
        }
    }

    public <O extends OutputStream, E> void createArchive(ArchiveJob job, HttpServletRequest servletRequest) {

        jobDAO.updateJobPhase(ExecutionPhase.EXECUTING, job.getJobId());

        LOG.trace("Started archive job " + job.getJobId());

        try {
            // TODO: check total size limit
            File archiveFile = getArchiveFile(job);

            List<ArchiveEntryDescriptor> entryDescriptors = job.getEntryDescriptors();

            // support directory used to generate folder inside tar files (path is redefined each time by TarEntry class)
            File supportDir = Files.createTempDirectory("dir").toFile();

            try (ArchiveHandler<O, E> handler = getArchiveHandler(archiveFile, job.getType())) {
                fillArchive(entryDescriptors, supportDir,
                        job.getPrincipal(), servletRequest, handler);
            } finally {
                FileSystemUtils.deleteRecursively(supportDir);
            }

        } catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    private <O extends OutputStream, E> void fillArchive(
            List<ArchiveEntryDescriptor> entryDescriptors,
            File supportDir, TokenPrincipal tokenPrincipal,
            HttpServletRequest servletRequest, ArchiveHandler<O, E> handler) throws IOException {

        List<ArchiveEntryDescriptor> noTargetEntryDescriptors
                = entryDescriptors.stream().filter(ed -> !ed.isPointingToAnotherNode())
                        .collect(Collectors.toList());

        // Start with archive entry descriptors which don't point to another node
        List<String> vosPaths = noTargetEntryDescriptors.stream()
                .map(ed -> ed.getVosPath())
                .collect(Collectors.toList());
        
        String commonParent = getCommonParent(entryDescriptors);

        if (!vosPaths.isEmpty()) {
            for (FileInfo fileInfo : fileDAO.getArchiveFileInfos(vosPaths)) {

                String relPath = fileInfo.getVirtualPath().substring(commonParent.length());

                this.insertEntryIntoArchive(fileInfo, supportDir, relPath, tokenPrincipal, servletRequest, handler);

            }
        }

        List<ArchiveEntryDescriptor> pointingEntryDescriptors
                = entryDescriptors.stream().filter(ed -> ed.isPointingToAnotherNode())
                        .collect(Collectors.toList());

        // Now archive entry descriptors pointing to another node
        List<String> targetNodesVosPaths = pointingEntryDescriptors.stream()
                .map(ed -> ed.getTargetNodeVosPath())
                .collect(Collectors.toList());

        if (!targetNodesVosPaths.isEmpty()) {
            for (FileInfo fileInfo : fileDAO.getArchiveFileInfos(targetNodesVosPaths)) {

                // relPaths is calculated from base node
                String targetNodeVosPath = fileInfo.getVirtualPath();
                List<String> linkVosPaths = pointingEntryDescriptors.stream()
                        .filter(ed -> ed.getTargetNodeVosPath().equals(targetNodeVosPath))
                        .map(ed -> ed.getVosPath())
                        .collect(Collectors.toList());

                for (String vosPath : linkVosPaths) {
                    String relPath = vosPath.substring(commonParent.length());

                    this.insertEntryIntoArchive(fileInfo, supportDir, relPath, tokenPrincipal, servletRequest, handler);
                }
            }
        }
    }

    private <O extends OutputStream, E> void insertEntryIntoArchive(
            FileInfo fileInfo, File supportDir, String relPath,
            TokenPrincipal tokenPrincipal, HttpServletRequest servletRequest, ArchiveHandler<O, E> handler)
            throws IOException {
        if (fileInfo.isDirectory()) {
            handler.putNextEntry(supportDir, relPath);
            return;
        }

        // I retrieve only external links
        // local links have been resolved before calling this endpoint
        // TODO: we need to discuss about internal links in container nodes
        if (fileInfo.isLink()) {
            String target = fileInfo.getTarget();
            if (!target.startsWith("vos://")) {
                downloadExternalLinkIntoArchive(fileInfo, relPath,
                        tokenPrincipal, handler, servletRequest);
            }
            return;
        }
        
        writeFileIntoArchive(fileInfo, relPath, tokenPrincipal, handler);

    }

    private File getArchiveFile(ArchiveJob job) throws IOException {

        File parentDir = getArchiveParentDir(job.getPrincipal());

        if (!parentDir.exists()) {
            if (!parentDir.mkdirs()) {
                LOG.error("Unable to create directory " + parentDir.getAbsolutePath());
                throw new InternalFaultException("Unable to create temporary directory for job");
            }
        }

        checkQuotaLimit(parentDir);

        File archiveFile = parentDir.toPath().resolve(job.getJobId() + "." + job.getType().getExtension()).toFile();

        if (!archiveFile.createNewFile()) {
            LOG.error("Unable to create file " + archiveFile.getAbsolutePath());
            throw new InternalFaultException("Unable to create archive file");
        }

        return archiveFile;
    }

    /**
     * If used working space exceeds quota limit throws an
     * InsufficientStorageException.
     */
    private void checkQuotaLimit(File parentDir) throws IOException {
        long usedSpace = Files.walk(parentDir.toPath()).mapToLong(p -> p.toFile().length()).sum();

        if (usedSpace > generatedDirMaxSize.toBytes()) {
            throw new QuotaExceededException("Archive size limit exceeded.");
        }
    }

    public File getArchiveParentDir(Principal principal) {
        return generatedDir.toPath().resolve(principal.getName()).toFile();
    }

    private String getCommonParent(List<ArchiveEntryDescriptor> entryDescriptors) {
        
        if(entryDescriptors.isEmpty()) {
            throw new IllegalArgumentException("Empty descriptors list");
        } else if(entryDescriptors.size() == 1) {
            return NodeUtils.getParentPath(entryDescriptors.get(0).getVosPath());
        }
        
        // Get list of parent paths
        List<String[]> vosParentPaths = entryDescriptors.stream().map(
                ed -> NodeUtils.getParentPath(ed.getVosPath()).split("/"))                
                .collect(Collectors.toList());
                
        // Get minimum size of split vosParentPaths arrays
        
        int minSize = vosParentPaths.stream()
                .mapToInt(v->v.length).min().orElseThrow();
                
        switch(minSize) {
            case 0:
                return "/";                
                
            case 1:
                // this should never happen
                throw new IllegalArgumentException("Invalid vosPath");
        }
        
        StringBuilder sb = new StringBuilder();
        
        for(int i = 1; i < minSize; i++) {
            List<String> elements = new ArrayList<>();
            for(String[] s : vosParentPaths) {
                elements.add(s[i]);
            }
            String sample = elements.get(0);
            if(elements.stream().allMatch(e->e.equals(sample)))
                sb.append("/").append(sample);
        }
                
        return sb.toString();
        
    }

    private abstract class ArchiveHandler<O extends OutputStream, E> implements AutoCloseable {

        private final O os;
        private final File parentDir;

        ArchiveHandler(O os, File parentDir) {
            this.os = os;
            this.parentDir = parentDir;
        }

        public abstract E getEntry(File file, String path);

        protected abstract void putNextEntry(E entry) throws IOException;

        public final void putNextEntry(File file, String path) throws IOException {
            putNextEntry(getEntry(file, path));
            checkQuotaLimit(parentDir);
        }

        public final O getOutputStream() {
            return os;
        }

        @Override
        public void close() throws IOException {
            os.close();
        }
    }

    private class TarArchiveHandler extends ArchiveHandler<TarOutputStream, TarEntry> {

        TarArchiveHandler(File archiveFile) throws IOException {
            super(new TarOutputStream(new BufferedOutputStream(new FileOutputStream(archiveFile))), archiveFile.getParentFile());
        }

        @Override
        public TarEntry getEntry(File file, String path) {
            return new TarEntry(file, path);
        }

        @Override
        public void putNextEntry(TarEntry entry) throws IOException {
            getOutputStream().putNextEntry(entry);
        }
    }

    private class ZipArchiveHandler extends ArchiveHandler<ZipOutputStream, ZipEntry> {

        ZipArchiveHandler(File archiveFile) throws IOException {
            super(new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(archiveFile))), archiveFile.getParentFile());
        }

        @Override
        public ZipEntry getEntry(File file, String path) {
            if (file.isDirectory()) {
                // ZipEntry assumes that paths ending with / are folders
                path += "/";
            }
            return new ZipEntry(path);
        }

        @Override
        public void putNextEntry(ZipEntry entry) throws IOException {
            getOutputStream().putNextEntry(entry);
        }
    }

    private ArchiveHandler getArchiveHandler(File archiveFile, ArchiveJob.Type type) throws IOException {
        switch (type) {
            case TAR:
                return new TarArchiveHandler(archiveFile);
            case ZIP:
                return new ZipArchiveHandler(archiveFile);
            default:
                throw new UnsupportedOperationException("Type " + type + " not supported yet");
        }
    }

    private <O extends OutputStream, E> void downloadFromUrlIntoArchive(String url, String relPath, TokenPrincipal tokenPrincipal, ArchiveHandler<O, E> handler) {
        LOG.trace("Downloading file from " + url);

        restTemplate.execute(url, HttpMethod.GET, req -> {
            HttpHeaders headers = req.getHeaders();
            if (tokenPrincipal.getToken() != null) {
                headers.setBearerAuth(tokenPrincipal.getToken());
            }
        }, res -> {
            File tmpFile = Files.createTempFile("download", null).toFile();
            try (FileOutputStream os = new FileOutputStream(tmpFile)) {
                res.getBody().transferTo(os);
                handler.putNextEntry(tmpFile, relPath);
                try (FileInputStream is = new FileInputStream(tmpFile)) {
                    is.transferTo(handler.getOutputStream());
                }
            } finally {
                tmpFile.delete();
            }
            return null;
        }, new Object[]{});
    }

    private <O extends OutputStream, E> void downloadExternalLinkIntoArchive(
            FileInfo fileInfo, String relPath, TokenPrincipal tokenPrincipal,
            ArchiveHandler<O, E> handler, HttpServletRequest servletRequest) {

        String url = fileInfo.getTarget();

        if (url == null || url.isBlank()) {
            LOG.error("Target URL of link at path: {} is null or blank", fileInfo.getVirtualPath());
            throw new InternalFaultException("Target URL of link at path: "
                    + fileInfo.getVirtualPath() + " is null or blank");
        }

        // Append token if url is recognized
        if (linkedServiceDAO.isLinkedServiceUrl(url)) {
            url += "?token=" + getEndpointToken(tokenPrincipal, url, servletRequest);
        }

        downloadFromUrlIntoArchive(url, relPath, tokenPrincipal, handler);

    }

    private <O extends OutputStream, E> void writeFileIntoArchive(FileInfo fileInfo, String relPath, TokenPrincipal tokenPrincipal, ArchiveHandler<O, E> handler) throws IOException {
        if (!authorizationService.isDownloadable(fileInfo, tokenPrincipal)) {
            throw PermissionDeniedException.forPath(fileInfo.getVirtualPath());
        }

        File file = new File(fileInfo.getFilePath());
        LOG.trace("Adding file " + file.getAbsolutePath() + " to tar archive");

        try (InputStream is = new FileInputStream(file)) {
            handler.putNextEntry(file, relPath);
            is.transferTo(handler.getOutputStream());
        }
    }

    private String getEndpointToken(TokenPrincipal tokenPrincipal,
            String endpoint, HttpServletRequest servletRequest) {

        String token = tokenPrincipal.getToken();

        if (token == null) {
            throw new PermissionDeniedException("Token is null");
        }

        TokenExchangeRequest exchangeRequest = new TokenExchangeRequest()
                .setSubjectToken(token)
                .setResource(endpoint);

        // TODO: add audience and scope
        return rapClient.exchangeToken(exchangeRequest, servletRequest);
    }

}
