/*
 * This file is part of vospace-file-service
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.transfer.persistence;

import it.inaf.ia2.transfer.persistence.model.FileInfo;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import javax.sql.DataSource;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.context.junit.jupiter.SpringExtension;
import org.springframework.test.util.ReflectionTestUtils;

@ExtendWith(SpringExtension.class)
@ContextConfiguration(classes = {DataSourceConfig.class})
@TestPropertySource(locations = "classpath:test.properties")
public class FileDAOTest {

    @Autowired
    private DataSource dataSource;

    @Value("${upload_location_id}")
    private int uploadLocationId;

    private FileDAO dao;

    @BeforeEach
    public void init() {
        dao = new FileDAO(dataSource);
        ReflectionTestUtils.setField(dao, "uploadLocationId", uploadLocationId);
    }

    @Test
    public void testGetBranchFileInfo() {
        List<FileInfo> fi = dao.getBranchFileInfos("/test100", "pippo");
        assertEquals(3, fi.size());
        List<FileInfo> fi2 = dao.getBranchFileInfos("/test100/test1001.txt", "pippo");
        assertEquals(1, fi2.size());
    }

    @Test
    public void testSetBranchLocationId() {
        dao.setBranchLocationId("/test100", "pippo", 3);
        List<FileInfo> fi = dao.getBranchFileInfos("/test100", "pippo");
        assertEquals(3, fi.size());

        fi.forEach(f -> {
            assertEquals(3, f.getLocationId());
        });

    }
    
    @Test
    public void testGetFileInfo() {

        Optional<FileInfo> optFileInfo = dao.getFileInfo("/test1/file1.txt");

        assertTrue(optFileInfo.isPresent());

        FileInfo fileInfo = optFileInfo.get();

        // TODO: refactor test
        //assertEquals("/home/username1/retrieve/file1.txt", fileInfo.getOsPath());
    }

    @Test
    public void testGetUnexistingFile() {
        assertTrue(dao.getFileInfo("/doesnt/exist").isEmpty());
    }

    @Test
    public void testGetArchiveFileInfos() {

        List<FileInfo> fileInfos = dao.getArchiveFileInfos(Arrays.asList("/public/file1", "/public/file2", "/public/subdir1"));

        assertEquals(6, fileInfos.size());

        assertEquals("/home/vospace/upload/user1/file1", fileInfos.get(0).getOsPath());
        assertEquals("/home/vospace/upload/user1/file2", fileInfos.get(1).getOsPath());
        assertEquals("user", fileInfos.get(1).getLocationType());
        assertTrue(fileInfos.get(2).isDirectory());
        assertNull(fileInfos.get(2).getLocationType());
        assertEquals("/home/username1/retrieve/subdir1/file3", fileInfos.get(3).getOsPath());
        assertEquals("/home/username1/retrieve/subdir1/file4", fileInfos.get(4).getOsPath());
        assertEquals("async", fileInfos.get(4).getLocationType());
        assertEquals("portal-file", fileInfos.get(5).getVirtualName());
        assertEquals("portal", fileInfos.get(5).getLocationType());
    }

    @Test
    public void testGetRemainingQuota() {

        assertEquals(600000l, dao.getRemainingQuota("/test_quota"));
        assertEquals(200000l, dao.getRemainingQuota("/test_quota/subdir"));
    }

    @Test
    public void testGetNoQuota() {

        assertNull(dao.getRemainingQuota("/"));
    }

    @Test
    public void testSetBusy() {

        FileInfo fileInfo = dao.getFileInfo("/public/file1").get();
        assertNull(fileInfo.getJobId());

        dao.setBusy(fileInfo.getVirtualPath(), "pippo1");
        assertEquals("pippo1", dao.getFileInfo("/public/file1").get().getJobId());

        dao.setBusy(fileInfo.getVirtualPath(), null);
        assertNull(dao.getFileInfo("/public/file1").get().getJobId());
    }

    @Test
    public void testReleaseNodesByJobId() {

        List<FileInfo> fi = dao.getBranchFileInfos("/test100", "pippo");
        assertEquals(3, fi.size());

        dao.releaseBusyNodesByJobId("pippo");

        fi = dao.getBranchFileInfos("/test100", "pippo");
        assertEquals(0, fi.size());

    }

    // TODO: refactor test
    /*
    @Test
    public void testSetOsName() {

        FileInfo fileInfo = dao.getFileInfo("/public/file1").get();
        assertTrue(fileInfo.getOsPath().endsWith("/file1"));

        dao.setOsName(fileInfo.getNodeId(), "file1-renamed");
        fileInfo = dao.getFileInfo("/public/file1").get();
        assertTrue(fileInfo.getOsPath().endsWith("/file1-renamed"));
    }*/

    @Test
    public void testUpdateFileAttributes() {

        FileInfo fileInfo = dao.getFileInfo("/public/file1").get();

        assertNull(fileInfo.getContentLength());
        assertNull(fileInfo.getContentType());
        assertNull(fileInfo.getContentEncoding());
        assertNull(fileInfo.getContentMd5());
        assertNull(fileInfo.getFsPath());

        dao.updateFileAttributes(fileInfo.getNodeId(), "/2021/09/24/UIID-1", "text/plain", "UTF-8", 50000l, "<md5>");

        fileInfo = dao.getFileInfo("/public/file1").get();
        assertEquals(50000l, fileInfo.getContentLength());
        assertEquals("2021/09/24/UIID-1", fileInfo.getFsPath());
        assertEquals("text/plain", fileInfo.getContentType());
        assertEquals("UTF-8", fileInfo.getContentEncoding());
        assertEquals("<md5>", fileInfo.getContentMd5());
    }
}
