/*
 * This file is part of vospace-file-service
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.transfer.controller;

import it.inaf.ia2.transfer.persistence.model.FileInfo;
import it.inaf.ia2.transfer.auth.TokenPrincipal;
import it.inaf.ia2.transfer.persistence.FileDAO;
import it.inaf.ia2.transfer.persistence.JobDAO;
import it.inaf.ia2.transfer.service.AuthorizationService;
import it.inaf.oats.vospace.exception.InvalidArgumentException;
import it.inaf.oats.vospace.exception.NodeNotFoundException;
import it.inaf.oats.vospace.exception.PermissionDeniedException;
import java.io.File;
import java.util.Optional;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class GetFileController extends FileController {

    private static final Logger LOG = LoggerFactory.getLogger(GetFileController.class);

    @Autowired
    private FileDAO fileDAO;

    @Autowired
    private JobDAO jobDAO;

    @Autowired
    private AuthorizationService authorizationService;

    @Autowired
    private HttpServletResponse response;

    @GetMapping("/**")
    public void getFile(@RequestParam(value = "jobId", required = false) String jobId) {

        String path = getPath();

        LOG.debug("getFile called for path {}", path);

        if (jobId == null) {
            LOG.debug("getFile called for path {}", path);
        } else {
            LOG.debug("getFile called for path {} with jobId {}", path, jobId);

            if (!jobDAO.isJobExisting(jobId)) {
                throw new InvalidArgumentException("Job " + jobId + " not found");
            }
        }

        handleFileJob(() -> {
            Optional<FileInfo> optFileInfo = fileDAO.getFileInfo(path);

            if (optFileInfo.isPresent()) {

                FileInfo fileInfo = optFileInfo.get();

                if (!authorizationService.isDownloadable(fileInfo, (TokenPrincipal) request.getUserPrincipal())) {
                    throw PermissionDeniedException.forPath(path);
                }

                File file = new File(fileInfo.getFilePath());
                FileResponseUtil.getFileResponse(response, file, path);
            } else {
                throw new NodeNotFoundException(path);
            }
        }, jobId);
    }
}
