/*
 * This file is part of vospace-file-service
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.transfer.service;

import it.inaf.ia2.transfer.auth.TokenPrincipal;
import it.inaf.ia2.transfer.persistence.model.FileInfo;
import it.inaf.ia2.transfer.persistence.FileDAO;
import it.inaf.ia2.transfer.persistence.JobDAO;
import java.io.File;
import java.nio.file.Path;
import java.util.List;
import org.apache.commons.io.FileUtils;
import org.assertj.core.util.Files;
import org.junit.jupiter.api.AfterEach;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.util.FileSystemUtils;

@SpringBootTest
@AutoConfigureMockMvc(addFilters = false)
public class CopyServiceTest {

    @MockBean
    private FileDAO fileDao;

    @MockBean
    private JobDAO jobDao;

    @Autowired
    private MockMvc mockMvc;
    
    @Autowired
    private FileCopyService fileCopyService;

    private static File sourceDirectory;
    private static File destinationDirectory;
    private static File sourceFile;
    private static String jobId;
    private static String sourceVosRootPath;
    private static String destVosRootPath;
    private static String filename;
    private static String userId;

    @BeforeAll
    public static void setUp() {
        jobId = "pippo_copy";
        userId = "user_copy";
        
        filename = "file.txt";
        sourceVosRootPath = "/source/" + filename;
        destVosRootPath = "/destination/" + filename;
    }
    
    @BeforeEach
    public void setUpPerTest() throws Exception {
        sourceDirectory = Files.newTemporaryFolder();
        destinationDirectory = Files.newTemporaryFolder();
        sourceFile = this.createFile(sourceDirectory, filename);
    }

    @AfterEach
    public void tearDown() {
        FileSystemUtils.deleteRecursively(sourceDirectory);
        FileSystemUtils.deleteRecursively(destinationDirectory);
    }

    @Test
    public void testCopyService() throws Exception {
        
        when(fileDao.getBranchFileInfos(eq(sourceVosRootPath), eq(jobId)))
                .thenReturn(
                        List.of(this.createBaseFileInfo(
                                filename, sourceDirectory, sourceVosRootPath)));
        FileInfo destinationFileInfo = this.createBaseDestinationFileInfo(
                                filename, destinationDirectory, destVosRootPath);
        
        when(fileDao.getBranchFileInfos(eq(destVosRootPath), eq(jobId)))
                .thenReturn(List.of(destinationFileInfo));
        
        when(fileDao.getRemainingQuota(any())).thenReturn(null);
                
        assertTrue(sourceFile.exists());
   
        fileCopyService.copyFiles(
                sourceVosRootPath, destVosRootPath, jobId, fakePrincipal(userId));
                
        assertTrue(sourceFile.exists());        
        File destinationFile = Path.of(destinationFileInfo.getFilePath()).toFile();
        assertTrue(destinationFile.exists());               
        assertTrue(FileUtils.contentEquals(sourceFile, destinationFile));

    }

    private FileInfo createBaseFileInfo(String fileName, File directory, String vosPath) {
        FileInfo fileInfo = new FileInfo();
        fileInfo.setActualBasePath("/");
        fileInfo.setFsPath(getTestFilePath(fileName, directory));
        fileInfo.setVirtualPath(vosPath);
        fileInfo.setPublic(false);
        fileInfo.setOwnerId(userId);
        
        return fileInfo;
    }
    
    private FileInfo createBaseDestinationFileInfo(String fileName, File directory, String vosPath) {
        FileInfo fileInfo = new FileInfo();
        fileInfo.setActualBasePath(directory.getAbsolutePath());
        fileInfo.setVirtualPath(vosPath);
        fileInfo.setPublic(false);
        fileInfo.setOwnerId(userId);
        
        return fileInfo;
    }

    private String getTestFilePath(String fileName, File directory) {
        return directory.toPath().resolve(fileName).toFile().getAbsolutePath();
    }
     
    private File createFile(File parent, String filename) throws Exception {
        parent.mkdir();
        
        File file = parent.toPath().resolve(filename).toFile();
        file.createNewFile();
        java.nio.file.Files.write(file.toPath(), "content_of_file".getBytes());
        
        return file;
    }
    
    private TokenPrincipal fakePrincipal(String name) {
        TokenPrincipal principal = mock(TokenPrincipal.class);
        when(principal.getName()).thenReturn(name);
        when(principal.getToken()).thenReturn("any_token_not_null");
        return principal;
    }

}
