/*
 * This file is part of vospace-datamodel
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace.datamodel;

import java.util.ArrayList;
import java.util.List;
import net.ivoa.xml.vospace.v2.DataNode;
import net.ivoa.xml.vospace.v2.LinkNode;
import net.ivoa.xml.vospace.v2.Node;
import net.ivoa.xml.vospace.v2.Property;
import static org.junit.jupiter.api.Assertions.assertArrayEquals;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.Test;

public class NodeUtilsTest {

    @Test
    public void testGetPathFromRequestURLString() {

        String requestUrl = "http://localhost/vospace/nodes/a/b/c/";
        assertEquals("/a/b/c", NodeUtils.getPathFromRequestURLString(requestUrl));
    }

    @Test
    public void testGetPathWithSpacesFromRequestURLString() {

        String requestUrl = "http://localhost/vospace/nodes/a/b/c%20d%20%C3%A4+%2B.pdf";
        assertEquals("/a/b/c d ä +.pdf", NodeUtils.getPathFromRequestURLString(requestUrl));
    }

    @Test
    public void testEncodePathSpecialChars() {

        String specialChars = "ä è#+ /other/+-ò@";
        assertEquals("%C3%A4%20%C3%A8%23%2B%20/other/%2B-%C3%B2%40", NodeUtils.urlEncodePath(specialChars));
    }

    @Test
    public void testIllegalBrakets() {
        testIllegalChars("<no>.pdf");
    }

    @Test
    public void testIllegalQuestionMark() {
        testIllegalChars("???.pdf");
    }

    @Test
    public void testIllegalQuotes() {
        testIllegalChars("\"'.pdf");
    }

    @Test
    public void testIllegalSlashEncoded() {
        testIllegalChars("%2F.pdf");
    }

    private void testIllegalChars(String illegalString) {
        boolean exception = false;
        try {
            NodeUtils.getPathFromRequestURLString("http://localhost/vospace/nodes/path/to/" + illegalString);
        } catch (IllegalArgumentException ex) {
            exception = true;
        }
        assertTrue(exception);
    }

    @Test
    public void getPathFromRequestURLStringTest() {

        String urlForTest = "http://server.example.com/vospace/";
        String result = NodeUtils.getPathFromRequestURLString(urlForTest);
        assertEquals("/", result);

        urlForTest = "http://server.example.com/vospace/nodes";
        result = NodeUtils.getPathFromRequestURLString(urlForTest);
        assertEquals("/", result);

        urlForTest = "http://server.example.com/vospace/nodes/mydata";
        result = NodeUtils.getPathFromRequestURLString(urlForTest);
        assertEquals("/mydata", result);

        urlForTest = "http://server.example.com/vospace/nodes/mydata1/uffi/nonso/pappa.txt";
        result = NodeUtils.getPathFromRequestURLString(urlForTest);
        assertEquals("/mydata1/uffi/nonso/pappa.txt", result);
    }

    // Is it a possible case?
    @Test
    public void subPathComponentsTest1() {
        String pathForTest = "";
        List<String> result = NodeUtils.subPathComponents(pathForTest);
        assertTrue(result.isEmpty());
    }

    @Test
    public void subPathComponentsTest2() {

        String pathForTest = "/";
        List<String> result = NodeUtils.subPathComponents(pathForTest);
        List<String> expected = new ArrayList<>();
        expected.add("/");
        assertArrayEquals(expected.toArray(), result.toArray());
    }

    @Test
    public void subPathComponentsTest3() {

        String pathForTest = "/mynode1";
        List<String> result = NodeUtils.subPathComponents(pathForTest);
        List<String> expected = new ArrayList<>();
        expected.add("/mynode1");
        assertArrayEquals(expected.toArray(), result.toArray());
    }

    @Test
    public void subPathComponentsTest4() {

        String pathForTest = "/mydata1/uffi/nonso/pappa.txt";
        List<String> result = NodeUtils.subPathComponents(pathForTest);
        List<String> expected = new ArrayList<>();
        expected.add("/mydata1");
        expected.add("/mydata1/uffi");
        expected.add("/mydata1/uffi/nonso");
        expected.add("/mydata1/uffi/nonso/pappa.txt");
        assertArrayEquals(expected.toArray(), result.toArray());
    }

    @Test
    public void testCheckReadablePublicNode() {

        Property creator = getProperty(NodeProperties.CREATOR_URI, "user1");
        Property publicRead = getProperty(NodeProperties.PUBLIC_READ_URI, "true");

        Node node = new DataNode();
        node.getProperties().add(creator);
        node.getProperties().add(publicRead);

        assertTrue(NodeUtils.checkIfReadable(node, "user3", List.of()));
    }

    @Test
    public void testCheckReadableCreatorOnly() {

        Property creator = getProperty(NodeProperties.CREATOR_URI, "user1");

        Node node = new DataNode();
        node.getProperties().add(creator);

        assertTrue(NodeUtils.checkIfReadable(node, "user1", List.of()));
    }

    @Test
    public void testCheckReadableGroupsTrue() {

        Property creator = getProperty(NodeProperties.CREATOR_URI, "user1");
        Property groupWrite = getProperty(NodeProperties.GROUP_READ_URI, "group3 group4");

        Node node = new DataNode();
        node.getProperties().add(creator);
        node.getProperties().add(groupWrite);

        assertTrue(NodeUtils.checkIfReadable(node, "user2", List.of("group2", "group3")));
    }
    
    @Test
    public void testGetLastPathElement() {
        
        assertThrows(IllegalArgumentException.class,() -> 
        { NodeUtils.getLastPathElement(null);} );
                
        assertThrows(IllegalArgumentException.class,() -> 
        { NodeUtils.getLastPathElement("  ");} );
                
        assertThrows(IllegalArgumentException.class,() -> 
        { NodeUtils.getLastPathElement("http://asdasd");} );
        
        assertEquals("/", NodeUtils.getLastPathElement("/"));
        
        assertEquals("f1", NodeUtils.getLastPathElement("/pippo/test/f1"));
        
    }

    @Test
    public void testCheckReadableGroupsFalse() {

        Property creator = getProperty(NodeProperties.CREATOR_URI, "user1");
        Property groupWrite = getProperty(NodeProperties.GROUP_READ_URI, "group3 group4");

        Node node = new DataNode();
        node.getProperties().add(creator);
        node.getProperties().add(groupWrite);

        assertFalse(NodeUtils.checkIfReadable(node, "user2", List.of("group5", "group6")));
    }

    @Test
    public void testCheckWritableCreatorOnly() {

        Property creator = getProperty(NodeProperties.CREATOR_URI, "user1");

        Node node = new DataNode();
        node.getProperties().add(creator);

        assertTrue(NodeUtils.checkIfWritable(node, "user1", List.of()));
    }

    @Test
    public void testCheckWritableGroupsTrue() {

        Property creator = getProperty(NodeProperties.CREATOR_URI, "user1");
        Property groupWrite = getProperty(NodeProperties.GROUP_WRITE_URI, "group3 group4");

        Node node = new DataNode();
        node.getProperties().add(creator);
        node.getProperties().add(groupWrite);

        assertTrue(NodeUtils.checkIfWritable(node, "user2", List.of("group2", "group3")));
    }

    @Test
    public void testCheckWritableGroupsFalse() {

        Property creator = getProperty(NodeProperties.CREATOR_URI, "user1");
        Property groupWrite = getProperty(NodeProperties.GROUP_WRITE_URI, "group3 group4");

        Node node = new DataNode();
        node.getProperties().add(creator);
        node.getProperties().add(groupWrite);

        assertFalse(NodeUtils.checkIfWritable(node, "user2", List.of("group5", "group6")));
    }

    @Test
    public void testCheckWritableNoGroups() {

        Property creator = getProperty(NodeProperties.CREATOR_URI, "user1");

        Node node = new DataNode();
        node.getProperties().add(creator);

        assertFalse(NodeUtils.checkIfWritable(node, "user2", List.of()));
    }

    @Test
    public void testGetVosPath() {

        Node node = new DataNode();
        node.setUri("vos://example.com!vospace/mynode/child1/child2");

        assertEquals("/mynode/child1/child2", NodeUtils.getVosPath(node));
    }

    @Test
    public void testGetParentPath() {
        assertEquals("/node1/node2", NodeUtils.getParentPath("/node1/node2/node2"));
    }

    @Test
    public void testIsBusy() {
        DataNode node = new DataNode();
        node.setBusy(true);
        assertTrue(NodeUtils.getIsBusy(node));
    }

    @Test
    public void testIsNeverBusy() {
        LinkNode node = new LinkNode();
        assertFalse(NodeUtils.getIsBusy(node));
    }

    @Test
    public void testGetNodeName() {
        Node node = new DataNode();
        node.setUri("vos://example.com!vospace/mynode/child1/child2");
        assertEquals("child2", NodeUtils.getNodeName(node));
    }

    private Property getProperty(String uri, String value) {
        Property property = new Property();
        property.setUri(uri);
        property.setValue(value);
        return property;
    }
}
