package net.ivoa.xml.uws.v1;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.StringReader;
import java.io.StringWriter;
import javax.xml.bind.JAXB;
import net.ivoa.xml.uws.v1.JobSummary.JobInfo;
import net.ivoa.xml.vospace.v2.Protocol;
import net.ivoa.xml.vospace.v2.Transfer;
import org.junit.jupiter.api.Test;
import static org.junit.jupiter.api.Assertions.*;

public class JobSummaryTest {

    private static final ObjectMapper MAPPER = new ObjectMapper();

    @Test
    public void testXmlSerialization() throws Exception {

        JobSummary job = getJobSummary();

        String xml;
        try ( StringWriter sw = new StringWriter()) {
            JAXB.marshal(job, sw);
            xml = sw.toString();
            System.out.println(xml);
        }

        assertTrue(xml.contains("<uws:job"));
        assertTrue(xml.contains("<uws:jobInfo"));
        assertTrue(xml.contains("<vos:transfer"));

        try ( StringReader sr = new StringReader(xml)) {
            JobSummary deserialized = JAXB.unmarshal(sr, JobSummary.class);
            verifyJobsAreEquals(deserialized);
        }
    }

    @Test
    public void testJsonSerialization() throws Exception {

        JobSummary job = getJobSummary();

        String json = MAPPER.writeValueAsString(job);
        System.out.println(json);

        JobSummary deserialized = MAPPER.readValue(json, JobSummary.class);

        verifyJobsAreEquals(deserialized);
    }

    private JobSummary getJobSummary() {

        JobSummary job = new JobSummary();
        job.setJobId("job_id");
        job.setPhase(ExecutionPhase.PENDING);

        JobInfo jobInfo = new JobInfo();

        Transfer transfer = new Transfer();
        transfer.setVersion("2.1");
        transfer.setTarget("vos://example.com!vospace/mydata1");
        transfer.setDirection("pullFromVoSpace");
        Protocol protocol = new Protocol();
        protocol.setUri("ivo://ivoa.net/vospace/core#httpget");
        transfer.getProtocol().add(protocol);

        jobInfo.getAny().add(transfer);

        job.setJobInfo(jobInfo);

        return job;
    }

    private void verifyJobsAreEquals(JobSummary deserializedJob) {

        assertEquals("job_id", deserializedJob.getJobId());
        assertEquals(ExecutionPhase.PENDING, deserializedJob.getPhase());

        Transfer transfer = (Transfer) deserializedJob.getJobInfo().getAny().get(0);
        assertEquals("2.1", transfer.getVersion());
        assertEquals("pullFromVoSpace", transfer.getDirection());
        assertEquals("vos://example.com!vospace/mydata1", transfer.getTarget());

        Protocol protocol = transfer.getProtocol().get(0);
        assertEquals("ivo://ivoa.net/vospace/core#httpget", protocol.getUri());
    }
}
