from django.db import models
from composite_field import CompositeField

# Create your models here.

IMAGE_CATEGORY = (
  'SCIENCE',
  'CALIBRATION',
  'SIMULATION'
)

IMAGE_FIRST_GROUP = (
  'OBJECT',
  'STD',
  'BIAS',
  'DARK',
  'FLAT',
  'LINEARITY',
  'OTHER'
)


IMAGE_SECOND_GROUP = (
  'SKY',
  'LAMP',
  'DOME',
  'OTHER'
)

class ImageType(CompositeField):
  
  category = models.CharField(
    max_length=20,
    choices=[(d, d) for d in IMAGE_CATEGORY]
  )
  
  firstType = models.CharField(
    max_length=20,
    choices=[(d,d) for d in IMAGE_FIRST_GROUP]
  )
  
  secondType = models.CharField(
    max_length=20,
    choices=[(d,d) for d in IMAGE_SECOND_GROUP]
  )
  

class ImageStatistics(models.Model):
  min = models.FloatField()
  max = models.FloatField()
  mean = models.FloatField()
  stddev = models.FloatField()
  median = models.FloatField()

  
class ImageBaseFrame(models.Model):
  exposureTime = models.FloatField()
  imgNumber = models.PositiveSmallIntegerField()
  naxis1 = models.PositiveIntegerField()
  naxis2 = models.PositiveIntegerField()
  imageType = ImageType()
  stats = models.OneToOneField(
    ImageStatistics,
    models.SET_NULL,
    blank=True,
    null=True,
  )
  
  class Meta:
    abstract = True
    

class Instrument(models.Model):
  instrumentName = models.CharField(max_length=100)
  telescopeName = models.CharField(max_length=100)
  
  class Meta:
    unique_together = (('instrumentName', 'telescopeName'))


class Pointing(CompositeField):
  rightAscension = models.FloatField()
  declination = models.FloatField()
  orientation = models.FloatField()


class ImageSpaceFrame(ImageBaseFrame):
  observationDateTime = models.DateTimeField()
  observationId = models.PositiveIntegerField()
  ditherNumber = models.PositiveSmallIntegerField()
  instrument = models.ForeignKey(Instrument, 
                                 on_delete=models.CASCADE,
                                 related_name='+')
  commandedPointing = Pointing()

  class Meta:
    abstract = True


NISP_DETECTOR_ID = (
  '11','12','13','14',
  '21','22','23','24',
  '31','32','33','34',
  '41','42','43','44'
)


class NispDetector(models.Model):
  detectorId = models.CharField(
    max_length=2,
    choices = [(d,d) for d in NISP_DETECTOR_ID]
  )
  gain = models.FloatField()
  readoutNoise = models.FloatField()
  rawFrame = models.ForeignKey('NispRawFrame',
                               related_name='detectors',
                               on_delete=models.CASCADE)
  class Meta:
    unique_together = (('detectorId', 'rawFrame'))


WCS_COORDINATE_TYPE = (
  'RA',
  'DEC',
  'GLON',
  'GLAT',
  'ELON',
  'ELAT'
)

WCS_PROJECTION_TYPE = (
  'LOG',
  'TAN',
  'SIN'
)


class CtypeWcs(CompositeField):
  coordinateType = models.CharField(
    max_length=4,
    choices = [(d,d) for d in WCS_COORDINATE_TYPE]
  )
  projectionType = models.CharField(
    max_length=3,
    choices = [(d,d) for d in WCS_PROJECTION_TYPE]
  )


class Astrometry(models.Model):
  ctype1 = CtypeWcs()
  ctype2 = CtypeWcs()
  crval1 = models.FloatField()
  crval2 = models.FloatField()
  crpix1 = models.FloatField()
  crpix2 = models.FloatField()
  cd1_1 = models.FloatField()
  cd1_2 = models.FloatField()
  cd2_1 = models.FloatField()
  cd2_2 = models.FloatField()
  detector = models.OneToOneField(NispDetector,
                                  related_name='astrometry',
                                  blank=True,
                                  null=True,
                                  on_delete=models.CASCADE)


NISP_FILTER_WHEEL = (
  'Y',
  'J',
  'H',
  'OPEN',
  'CLOSE'
)

NISP_GRISM_WHEEL = (
  'BLUE0',
  'RED0',
  'RED90',
  'RED180'
  'OPEN'
  'CLOSE'
)


class DataContainer(models.Model):
  fileFormat = models.CharField(
    max_length=10
  )
  formatIdentifier = models.CharField(
    max_length=20
  )
  formatVersion = models.CharField(
    max_length=20
  )
  url = models.URLField()  
  
  

class NispRawFrame(ImageSpaceFrame):
  filterWheelPosition = models.CharField(
    max_length=15,
    choices = [(d,d) for d in NISP_FILTER_WHEEL]
  )

  grismWheelPosition = models.CharField(
    max_length=15,
    choices = [(d,d) for d in NISP_GRISM_WHEEL]
  )
  frameFile = models.OneToOneField(DataContainer,
                                   on_delete=models.CASCADE)
  

