# Copyright (C) 2020 INAF
# This software is distributed under the terms of the BSD-3-Clause license
#
# Authors:
#    Bulgarelli Andrea <andrea.bulgarelli@inaf.it>
#    Baroncelli Leonardo <leonardo.baroncelli@inaf.it>
#    Parmiggiani Nicolò <nicolo.parmiggiani@inaf.it>
#    Addis Antonio <antonio.addis@inaf.it>

import json
import random
import subprocess
from time import sleep
from os import listdir
from datetime import datetime 
from os.path import isfile, join

import SAG_MODULE
import SAG_MODULE__POA

import SAG_CUSTOM_TYPES_MODULE
import SAG_CUSTOM_TYPES_MODULE__POA

from Acspy.Servants.ACSComponent import ACSComponent
from Acspy.Servants.ComponentLifecycle import ComponentLifecycle
from Acspy.Servants.ContainerServices import ContainerServices

import SAGErr
import SAGErrImpl


class STATUS:
    """
    The status of the SAGReco component.
    """
    NOT_STARTED = 0
    READY = 1
    PROCESSING = 2

class SAGRecoImpl(SAG_MODULE__POA.SAGReco, ACSComponent, ComponentLifecycle, ContainerServices):
    
    """
    This interface is used to start RECO analysis processes. It is used internally within the SAG system.
    """
    
    def __init__(self):
        ACSComponent.__init__(self)
        ContainerServices.__init__(self)
        self._logger = self.getLogger()
        self.sbID = None
        self.process = None
        self.scriptsDir = None
        self.outputDir = None
        self.status = STATUS.NOT_STARTED


    def initialize(self):
        #Assign variable values
        #Initialize data
        pass

    def execute(self):
        #Retrieve components
        self.sagDatabase = self.getComponent("SIM_SAG_DATABASE")

    def cleanUp(self):
        pass

    def aboutToAbort(self):
        #Do any critical clean up
        #Continue with less critical stuff such as releasing components and other activities similar to cleanUp
        pass


    def initializeWithData(self, sbID):
        """
        Initialize the scheduling block ID of the SAGReco component.
        """
        self.sbID=sbID
        self.scriptsDir = self.sagDatabase.getScriptsDir()
        self.outputDir = self.sagDatabase.getPipelineOutputDir()
        inputDataRootDir = self.sagDatabase.getSubArraysInputDataRootDir()
        self.inputDataDir = join(inputDataRootDir, "datadir_"+self.sbID)


    def start_sim_pipe(self):
        """
        Start a dummy implementation of the reco analysis process.
        """
        cmd = "bash "+self.scriptsDir+"/sim_pipe_start.sh "+self.sbID+" "+self.outputDir+" reco"
        self.process = subprocess.Popen(cmd, shell=True)
        self._logger.logInfo("[SAGRecoImpl - start_sim_pipe] cmd="+cmd+" Process pid: "+str(self.process.pid))

    def start_sim_data_processing_pipe(self):
        """
        Start a dummy implementation of the reco analysis process that consume data.
        """
        cmd = "bash "+self.scriptsDir+"/sim_pipe_data_processing.sh "+self.sbID+" "+self.outputDir+" reco "+self.inputDataDir
        self.process = subprocess.Popen(cmd, shell=True)
        self._logger.logInfo("[SAGRecoImpl - start_sim_data_processing_pipe] cmd="+cmd+" Process pid: "+str(self.process.pid))




    def start(self):
        """
        Start the reco analysis process.
        """
        self._logger.logInfo("[SAGRecoImpl - start]")
        self.start_sim_pipe()
        self.status = STATUS.READY


    def stop(self, sagCallback):
        """
        [Async] Tell the reco process to stop the analysis. The callback status is set to DONE when all the data receveid until the stop() call is processed.
        """
        self._logger.logInfo("[SAGRecoImpl - stop]")

        data = {}
        sagCallback.working(json.dumps(data))

        if self.status == STATUS.NOT_STARTED:
            self._logger.logInfo("[SAGRecoImpl - stop] Status was 'NOT_STARTED' no process to be killed")
            sagCallback.done(json.dumps({}))

        elif self.status == STATUS.READY:
            self.process.kill()
            self._logger.logInfo("[SAGRecoImpl - stop] Status was 'READY' - sbID="+str(self.sbID)+" process killed")
            self.status = STATUS.NOT_STARTED
            sagCallback.done(json.dumps({}))

        elif self.status == STATUS.PROCESSING:

            # find which are the last files to be processed for each input-data-dir
            filesToBeProcessed = set([join(self.inputDataDir, f) for f in listdir(self.inputDataDir) if isfile(join(self.inputDataDir, f))])
            self._logger.logInfo(f"[SAGRecoImpl - stop] Waiting for files to be processed in {self.inputDataDir}. Number of files: {len(filesToBeProcessed)}")

            data = {"filesRemaining":len(filesToBeProcessed)}
            sagCallback.working(json.dumps(data))
            
            while True:
                sleep(0.5)
                # monitor the directory
                currentfiles = set([join(self.inputDataDir, f) for f in listdir(self.inputDataDir) if isfile(join(self.inputDataDir, f))])
                filesToBeProcessed = filesToBeProcessed - (filesToBeProcessed - currentfiles) 
                self._logger.logInfo(f"[SAGRecoImpl - stop] Number of files: {len(filesToBeProcessed)}")

                data = {"filesRemaining":len(filesToBeProcessed)}
                sagCallback.working(json.dumps(data))

                # when the last files are consumed stop the processes
                if len(filesToBeProcessed) == 0:
                    break   

            self._logger.logInfo("[SAGRecoImpl - stop] All files have been processed! Killing the process..")

            self.process.kill()
            self._logger.logInfo("[SAGRecoImpl - stop] Status was 'PROCESSING' - sbID="+str(self.sbID)+" process killed")
            self.status = STATUS.NOT_STARTED
            data = {"filesRemaining":0}
            sagCallback.done(json.dumps(data))

            self._logger.logInfo("[SAGRecoImpl - stop] Returning...")

    def startDataProcessing(self):
        """
        Tell the reco analysis process to start to consume data.
        """
        self._logger.logInfo("[SAGRecoImpl - startDataProcessing]")

        if self.status == STATUS.PROCESSING:
            raise SAGErrImpl.SAGDataProcessingAlreadyStartedExImpl()

        elif self.status == STATUS.READY:
            self.process.kill()
            self.start_sim_data_processing_pipe()
            self.status = STATUS.PROCESSING

        elif self.status == STATUS.NOT_STARTED:
            raise SAGErrImpl.SAGDataProcessingCannotStartExImpl()


    def stopDataProcessing(self):
        """
        Tell the reco analysis process to stop to consume data.
        """
        self._logger.logInfo("[SAGRecoImpl - stopDataProcessing]")
        
        try:
            if self.status == STATUS.PROCESSING:
                self.process.kill()
                self.start_sim_pipe()
                self.status = STATUS.READY
                self._logger.logInfo("[SAGRecoImpl - stopDataProcessing] sbID="+self.sbID+" process killed")
        
            elif self.status == STATUS.READY or self.status == STATUS.NOT_STARTED:
                raise SAGErrImpl.SAGDataProcessingAlreadyStoppedExImpl()
        
        except Exception as e:
            self._logger.logInfo("[SAGRecoImpl - stopDataProcessing] Exception!",str(e))

        self._logger.logInfo("[SAGRecoImpl - stopDataProcessing] finish!")

    def getMonitoringInfo(self):
        """
        Return monitoring data of reco analysis.
        """
        self._logger.logInfo("[SAGRecoImpl - getMonitoringInfo]")
        monitoredParamsList = []
        monitoredParamsList.append(SAG_CUSTOM_TYPES_MODULE.MonitoredParam("status", self.status))
        monitoredParamsList.append(SAG_CUSTOM_TYPES_MODULE.MonitoredParam("timestamp", datetime.now().timestamp()))
        monitoredParamsList.append(SAG_CUSTOM_TYPES_MODULE.MonitoredParam("schedulingBlockID", int(self.sbID)))
        monitoredParamsList.append(SAG_CUSTOM_TYPES_MODULE.MonitoredParam("inputDataRate", random.random()*100))
        monitoredParamsList.append(SAG_CUSTOM_TYPES_MODULE.MonitoredParam("inputEventRate", random.random()*100))
        monitoredParamsList.append(SAG_CUSTOM_TYPES_MODULE.MonitoredParam("recoStepOneMeanProcessingRate", random.random()*100))
        monitoredParamsList.append(SAG_CUSTOM_TYPES_MODULE.MonitoredParam("recoStepTwoMeanProcessingRate", random.random()*100))
        return monitoredParamsList

