# Copyright (C) 2020 INAF
# This software is distributed under the terms of the BSD-3-Clause license
#
# Authors:
#    Bulgarelli Andrea <andrea.bulgarelli@inaf.it>
#    Baroncelli Leonardo <leonardo.baroncelli@inaf.it>
#    Parmiggiani Nicolò <nicolo.parmiggiani@inaf.it>
#    Addis Antonio <antonio.addis@inaf.it>

import SAG_MODULE
import SAG_MODULE__POA

import SAG_CUSTOM_TYPES_MODULE
import SAG_CUSTOM_TYPES_MODULE__POA

from Acspy.Servants.ACSComponent import ACSComponent
from Acspy.Servants.ComponentLifecycle import ComponentLifecycle
from Acspy.Servants.ContainerServices import ContainerServices

from Acspy.Nc.Consumer import Consumer
from Acspy.Nc.Supplier import Supplier

class SAGNotificationManagerImpl(SAG_MODULE__POA.SAGNotificationManager, ACSComponent, ContainerServices, ComponentLifecycle):
    """ 
    This interface defines methods to send/receive messages to/from external acada subsystems through ACS' notification channels. It is used internally within the SAG system.   
    """

    def __init__(self):
        ACSComponent.__init__(self)
        ContainerServices.__init__(self)
        self._logger = self.getLogger()

        #consumers
        self.mon_consumer = None
        self.adh_consumer = None
        self.cc_consumer = None
        self.rm_consumer = None
        self.hmi_consumer = None

        #suppliers
        self.mon_supplier = None
        self.adh_supplier = None
        self.cc_supplier = None
        self.rm_supplier = None
        self.hmi_supplier = None



    def initialize(self):
        """
        Create Consumer and Supplier objects for each acada subsystem.
        """
        #consumers
        self.mon_consumer = Consumer(SAG_CUSTOM_TYPES_MODULE.CHANNELNAME_MON)
        self.adh_consumer = Consumer(SAG_CUSTOM_TYPES_MODULE.CHANNELNAME_ADH)
        self.cc_consumer = Consumer(SAG_CUSTOM_TYPES_MODULE.CHANNELNAME_CC)
        self.rm_consumer = Consumer(SAG_CUSTOM_TYPES_MODULE.CHANNELNAME_RM)
        self.hmi_consumer = Consumer(SAG_CUSTOM_TYPES_MODULE.CHANNELNAME_HMI)

        #suppliers
        self.mon_supplier = Supplier(SAG_CUSTOM_TYPES_MODULE.CHANNELNAME_MON)
        self.adh_supplier = Supplier(SAG_CUSTOM_TYPES_MODULE.CHANNELNAME_ADH)
        self.cc_supplier = Supplier(SAG_CUSTOM_TYPES_MODULE.CHANNELNAME_CC)
        self.rm_supplier = Supplier(SAG_CUSTOM_TYPES_MODULE.CHANNELNAME_RM)
        self.hmi_supplier = Supplier(SAG_CUSTOM_TYPES_MODULE.CHANNELNAME_HMI)



    def mon_event_handler(self, event):
        # The event handler that will process messages coming from the MONINTORING subsytem.
        self._logger.logInfo(f"New message received from MON: {event}")

    def adh_event_handler(self, event):
        # The event handler that will process messages coming from the ARRAY DATA HANDLER subsytem.
        self._logger.logInfo(f"New message received from ADH: {event}")

    def cc_event_handler(self, event):
        # The event handler that will process messages coming from the CENTRAL CONTROL subsytem.
        self._logger.logInfo(f"New message received from CC: {event}")

    def rm_event_handler(self, event):
        # The event handler that will process messages coming from the RESOURCE MANAGER subsytem.
        self._logger.logInfo(f"New message received from RM: {event}")

    def hmi_event_handler(self, event):
        # The event handler that will process messages coming from the HUMAN MACHINE INTERFACE subsytem.
        self._logger.logInfo(f"New message received from HMI: {event}")


    def sendMessageToMON(self, event):
        # Send a message to the MONITORING subsytem.
        self._logger.logInfo(f"[SAGNotificationManagerImpl - sendMessageToMON] Publishing event {event}")
        self.mon_supplier.publishEvent(simple_data=event)

    def sendMessageToADH(self, event):
        # Send a message to the ARRAY DATA HANDLER subsytem.
        self._logger.logInfo(f"[SAGNotificationManagerImpl - sendMessageToADH] Publishing event {event}")
        self._logger.logInfo("Publishing event" + str(event))
        self.adh_supplier.publishEvent(simple_data=event)
    
    def sendMessageToCC(self, event):
        # Send a message to the CENTRAL CONTROL subsytem.
        self._logger.logInfo(f"[SAGNotificationManagerImpl - sendMessageToCC] Publishing event {event}")
        self._logger.logInfo("Publishing event" + str(event))
        self.cc_supplier.publishEvent(simple_data=event)
    
    def sendMessageToRM(self, event):
        # Send a message to the RESOURCE MANAGER subsytem.
        self._logger.logInfo(f"[SAGNotificationManagerImpl - sendMessageToRM] Publishing event {event}")
        self._logger.logInfo("Publishing event" + str(event))
        self.rm_supplier.publishEvent(simple_data=event)
    
    def sendMessageToHMI(self, event):
        # Send a message to the HUMAN MACHINE INTERFACE subsytem.
        self._logger.logInfo(f"[SAGNotificationManagerImpl - sendMessageToHMI] Publishing event {event}")
        self._logger.logInfo("Publishing event" + str(event))
        self.hmi_supplier.publishEvent(simple_data=event)


    def execute(self):
        """
        Define, for each Consumer object, the message type and the event handler that will process the message. 
        """
        self.mon_consumer.addSubscription(SAG_CUSTOM_TYPES_MODULE.ChannelEvent_FROM_MON, self.mon_event_handler)
        self.mon_consumer.consumerReady()

        self.adh_consumer.addSubscription(SAG_CUSTOM_TYPES_MODULE.ChannelEvent_FROM_ADH, self.adh_event_handler)
        self.adh_consumer.consumerReady()

        self.cc_consumer.addSubscription(SAG_CUSTOM_TYPES_MODULE.ChannelEvent_FROM_CC, self.cc_event_handler)
        self.cc_consumer.consumerReady()

        self.rm_consumer.addSubscription(SAG_CUSTOM_TYPES_MODULE.ChannelEvent_FROM_RM, self.rm_event_handler)
        self.rm_consumer.consumerReady()

        self.hmi_consumer.addSubscription(SAG_CUSTOM_TYPES_MODULE.ChannelEvent_FROM_HMI, self.hmi_event_handler)
        self.hmi_consumer.consumerReady()

    def cleanUp(self):
        """
        Disconnect all the Consumer objects. 
        """        
        self.mon_consumer.disconnect()
        self.adh_consumer.disconnect()
        self.cc_consumer.disconnect()
        self.rm_consumer.disconnect()
        self.hmi_consumer.disconnect()