# Copyright (C) 2020 INAF
# This software is distributed under the terms of the BSD-3-Clause license
#
# Authors:
#    Bulgarelli Andrea <andrea.bulgarelli@inaf.it>
#    Baroncelli Leonardo <leonardo.baroncelli@inaf.it>
#    Parmiggiani Nicolò <nicolo.parmiggiani@inaf.it>
#    Addis Antonio <antonio.addis@inaf.it>

from time import sleep
import json

from Acspy.Servants.ACSComponent import ACSComponent
from Acspy.Servants.ComponentLifecycle import ComponentLifecycle
from Acspy.Servants.ContainerServices import ContainerServices


import SAG_SUPERVISOR_MODULE
import SAG_SUPERVISOR_MODULE__POA

from SAG_IMPL_MODULE.SAGCallbackImpl import SAGCallbackImpl


from UTILITY_IMPL_MODULE.ContainerUtilityImpl import ContainerUtility 

from SAG_CUSTOM_TYPES_MODULE import INIT, WORKING, DONE, ERROR

import SAGErr
import SAGErrImpl

class SAGSupervisorImpl(SAG_SUPERVISOR_MODULE__POA.SAGSupervisor, ACSComponent, ComponentLifecycle, ContainerServices):
    
    """
    This interface is the entry point to the SAG. External components MUST use this interface to interact with the SAG.
    """

    def __init__(self):
        ACSComponent.__init__(self)
        ContainerServices.__init__(self)
        self._logger = self.getLogger()
        self.subArraysRefMap = {}
        self.containerUtility = None
        self.sagDatabase = None
        self.sagMonitor = None

    ## Lifecycle ##

    def initialize(self):
        self._logger.logInfo(f"[SAGSupervisorImpl - initialize]")
        self.containerUtility = ContainerUtility()

    def execute(self):
        self._logger.logInfo(f"[SAGSupervisorImpl - execute]")
        self.sagDatabase = self.getComponent("SIM_SAG_DATABASE")

    def cleanUp(self):
        self._logger.logInfo(f"[SAGSupervisorImpl - cleanUp]")
        self.releaseComponent("SIM_SAG_DATABASE")


    def aboutToAbort(self):
        #Do any critical clean up
        #Continue with less critical stuff such as releasing components and other activities similar to cleanUp
        pass
 
 
    """
    * [SUP-040] When a new SB is received, a dedicated SAG pipeline shall start in an automatic and controlled way.
    *
    * Implementation:
    * The SAGSupervisor receives the command to activate the SubArrayManager for a specific subArrayID. 
    * The SubArrayManager activates the SAGReco, SAGSci and SAGDQ components.
    * The SubArrayManager calls the "start" method of SAGReco, SAGSci and SAGDQ components to start the SAG pipeline processes.
    *
    * @raise SAGSchedulingBlockIdNotFoundInDatabase exception if no scheduling block with id=sbID is found in the database.
    * @raise SAGSubArrayManagerAlreadyActivated exception if the scheduling block is already being processed.
    * @raise SAGRecoAlreadyActivated exception if the SAGReco component is already activated.
    * @raise SAGDataQualityAlreadyActivated exception if the SAGDQ component is already activated. 
    * @raise SAGSciAlreadyActivated exception if the SAGSci component is already activated. 
    * @return void
    """
    def newSchedulingBlock(self, sbID):

        import pathlib
        self._logger.logInfo(f"[SAGSupervisorImpl - newSchedulingBlock] MYFOLDER {pathlib.Path(__file__).parent.absolute()}..")

        self.activateSubArrayManager(sbID)

        try:
            self.activateAndStartPipelines(sbID)
            
        except Exception as e:
            raise e    
        

    """
    * [SUP-110] SAG shall receive information when the execution of a Scheduling Blocks is cancelled. Analysis of the data taken until the cancellation command shall be executed.
    * 
    * Implementation:
    * The SAGSupervisor receives the command to stop the data processing (if it is already Activated) and to deactivate the SubArrayManager, SAGReco, SAGSci and SAGDQ components. 
    * The SAGSupervisor tells the SubArrayManager to stop the data processing: the SubArrayManager forwards the request to its childs components. SAGReco, SAGSci and SAGDQ kills the processes.
    * The SubArrayManager deactivates its childs components. 
    * The SAGSupervisor deactivates the SubArrayManager.
    *
    * @return void
    """
    def stopSchedulingBlock(self, sbID): #, callback):
        
        # callback = self.activateOffShoot(callback)

        # callback.working("")

        subArrayID = self.sagDatabase.getSubArrayIDFromSB(sbID)

        if subArrayID == -1:
            self._logger.logInfo("[SAGSupervisorImpl - stopSchedulingBlock] ERROR! subArrayID=-1..")
            # callback.err("SAGSchedulingBlockIdNotFoundInDatabaseExImpl")
            return

        if not self.stopAndDeactivatePipelines(subArrayID):# , callback)
            self._logger.logInfo("[SAGSupervisorImpl - stopSchedulingBlock] stopAndDeactivatePipelines returned False..")
            return 

        self._logger.logInfo("[SAGSupervisorImpl - stopSchedulingBlock] before calling deactivateSubArrayManager")

        if not self.deactivateSubArrayManager(subArrayID):# , callback)
            self._logger.logInfo("[SAGSupervisorImpl - stopSchedulingBlock] deactivateSubArrayManager returned False..")
            return

        # callback.done("")

    """
    * [SUP-50] When a sub-array enters in observing state, the SAG Supervisor shall inform the related SAG pipeline, and new data shall be received.
    *
    * Implementation: 
    * Start pipelines data acquisition for a Scheduling Block.

    * @param sbID: the id of the Scheduling Block.
    * @raise SAGSchedulingBlockIdNotFoundInDatabase exception if no scheduling block with id=sbID is found in the database.
    * @raise SAGDataProcessingAlreadyStarted exception if the data acquisition is already started.
    * @return void
    """
    def startObservation(self, sbID):
        subArrayID = self.sagDatabase.getSubArrayIDFromSB(sbID)

        if subArrayID == -1:
            raise SAGErrImpl.SAGSchedulingBlockIdNotFoundInDatabaseExImpl()

        # Check if SubArrayManager is active, raise Exception if it's not active!
        componentName = "PY_SUBARRAYMANAGER_"+str(subArrayID)
        status = self.containerUtility.checkIfActive(componentName)
        if status == 1:
            raise SAGErrImpl.SAGSubArrayManagerIsNotRunningExImpl()

        self.subArraysRefMap[subArrayID].startDataProcessing()


    """
    * [SUP-060] When a sub-array stops the observation, the SAG Supervisor shall inform SAG pipelines that shall shutdown data processing in an automatic and controlled way.

    * Implementation:
    * This method ask to the SubArrayManager to stop the data procesing for a specific Sub Array and Observing Block

    * @param sbID: the id of the Scheduling Block.
    * @raise SAGSchedulingBlockIdNotFoundInDatabase exception if no scheduling block with id=sbID is found in the database.
    * @raise SAGDataAcquisitionAlreadyStopped exception if the data acquisition is already stopped.
    * @return void
    """
    def stopObservation(self, sbID):
        
        subArrayID = self.sagDatabase.getSubArrayIDFromSB(sbID)

        if subArrayID == -1:
            raise SAGErrImpl.SAGSchedulingBlockIdNotFoundInDatabaseExImpl()

        # Check if SubArrayManager is active, raise Exception if it's not active!
        componentName = "PY_SUBARRAYMANAGER_"+str(subArrayID)
        status = self.containerUtility.checkIfActive(componentName)
        if status == 1:
            raise SAGErrImpl.SAGSubArrayManagerIsNotRunningExImpl()

        self.subArraysRefMap[subArrayID].stopDataProcessing()





    """
    * [Private]
    * Activate a SubArrayManager component.
    * The SAGSupervisor receives the command to activate the SubArrayManager for a specific subArrayID. 
    * The SAGSupervisor reads from the database subArrayID.
    * The SAGSupervisor acquires a reference of a SubArrayManager.
    * The SAGSupervisor inizializes the SubArrayManager component.
    * @raise SchedulingBlockIdNotFoundInDatabaseEx exception if no scheduling block with id=sbID is found in the database.
    * @raise SAGSubArrayManagerAlreadyActivated exception if the SubArrayManager component with id=sbID is already been Activated.
    * @return void
    """
    def activateSubArrayManager(self, sbID):

        subArrayID = self.sagDatabase.getSubArrayIDFromSB(sbID)

        if subArrayID == -1:
            raise SAGErrImpl.SAGSchedulingBlockIdNotFoundInDatabaseExImpl()

        componentName = "PY_SUBARRAYMANAGER_"+str(subArrayID)
        
        status = self.containerUtility.checkIfActive(componentName)
        self._logger.logInfo("[SAGSupervisorImpl - activateSubArrayManager] Status = "+str(status))

        if status == 2:
            raise SAGErrImpl.SAGSubArrayManagerAlreadyActivatedExImpl()

        sagSubArrayManagerRef = self.getDynamicComponent(   componentName, 
                                                            "IDL:sag/SAG_MODULE__POA/SAGSubArrayManager:1.0",
                                                            "SAG_IMPL_MODULE.SAGSubArrayManagerImpl",
                                                            "pyContainerForSubArrayManager_"+sbID
                                                        )

        self.subArraysRefMap[subArrayID] = sagSubArrayManagerRef
        
        sagSubArrayManagerRef.initializeWithData(sbID)
    



    """
    * [Private]
    * Deactivate a SubArrayManager component
    * The SAGSupervisor receives the command to deactivate the SubArrayManager for a specific subArrayID.
    * The SAGSupervisor reads from the database subArrayID.
    * The SAGSupervisor releases the reference of the SubArrayManager.
    * @raise SchedulingBlockIdNotFoundInDatabaseEx exception if no Sub Array is found in the database.
    * @raise SAGSubArrayManagerAlreadyDeactivated exception if the SubArrayManager component with id=sbID is already been Deactivated.
    * @return void
    """
    def deactivateSubArrayManager(self, subArrayID): #,callback):

        componentName = "PY_SUBARRAYMANAGER_"+str(subArrayID)
        
        status = self.containerUtility.checkIfActive(componentName)
        self._logger.logInfo("[SAGSupervisorImpl - deactivateSubArrayManager] Status = "+str(status))

        #if status == 0 or status == 1:
        #   callback.err("SAGSubArrayManagerAlreadyDeactivatedExImpl")
        #   return False

        del self.subArraysRefMap[subArrayID]
        
        self.releaseComponent(componentName)
    
        return True




    """
    * [Private]
    * The SAGSupervisor reads the subArrayID from the database.
    * The SAGSupervisor call the activateAndStartPipelines() method of the SubArrayManager.
    * @raise SchedulingBlockIdNotFoundInDatabase exception if no Sub Array is found in the database.
    * @raise SAGSubArrayManagerIsNotRunning exception if the specific SubArray manager is not running.
    * @retun void
    """
    def activateAndStartPipelines(self, sbID):

        subArrayID = self.sagDatabase.getSubArrayIDFromSB(sbID)

        if subArrayID == -1:
            raise SAGErrImpl.SAGSchedulingBlockIdNotFoundInDatabaseExImpl()

        componentName = "PY_SUBARRAYMANAGER_"+str(subArrayID)
        
        status = self.containerUtility.checkIfActive(componentName)
        self._logger.logInfo("[SAGSupervisorImpl - activateAndStartPipelines] Status = "+str(status))

        if status == 0 or status == 1:
            raise SAGErrImpl.SAGSubArrayManagerIsNotRunningExImpl()

        self.subArraysRefMap[subArrayID].activateAndStartPipelines()
    

    """
    * [Private]
    * The SAGSupervisor call the stopAndDeactivatePipelines() method of the SubArrayManager.
    * @return void
    """
    def stopAndDeactivatePipelines(self, subArrayID):# , callback):

        componentName = "PY_SUBARRAYMANAGER_"+str(subArrayID)
        
        status = self.containerUtility.checkIfActive(componentName)
        self._logger.logInfo("[SAGSupervisorImpl - stopAndDeactivatePipelines] Status = "+str(status))


        #if status == 0 or status == 1:
        #    callback.err("SAGSubArrayManagerIsNotRunningExImpl")
        #    return False


        myCallback = SAGCallbackImpl()

        cbObj = self.activateOffShoot(myCallback)

        self.subArraysRefMap[subArrayID].stopAndDeactivatePipelines(cbObj)

        while True:

            sleep(0.5)

            myCallbackInfo = myCallback.getInfo()
            self._logger.logInfo(f"[SAGSupervisorImpl - stopAndDeactivatePipelines] ==> waiting...myCallbackInfo: {myCallbackInfo}")

            cbStatus = myCallbackInfo.status
            cdData = json.loads(myCallbackInfo.data)

            
            if cbStatus == DONE:
                self._logger.logInfo(f"[SAGSupervisorImpl - stopAndDeactivatePipelines] Callback status: {cbStatus} Callback data: {cdData}")
                break
            
            if myCallbackInfo.status == ERROR:
                self._logger.logInfo(f"[SAGSupervisorImpl - stopAndDeactivatePipelines] Callback status: {cbStatus} Callback data: {cdData}")
                break


        return True