import os
import pytest
from time import sleep
from pathlib import Path

from sb import SchedulingBlockId 
from utils import countAliveProcessesByScriptName, \
                  simulateDataArrivalForSubArray

@pytest.mark.containers_name("aragornContainer,pyContainerForSubArrayManager_1,pyContainerForSchedulingBlock_1,pyContainerForSubArrayManager_2,pyContainerForSchedulingBlock_2")
class TestSAGSubArrayPipelinesSupervisor:


    @pytest.mark.test_name("test_sag_supervisor_activation")
    def test_sag_supervisor_activation(self, printTestName, acsEnvironment, acsContainers, acsPyClient):

        sagSupervisor = None

        try:
            sagSupervisor = acsPyClient.getComponent('PY_SAGSUPERVISOR')
        except Exception as ex:
            print("[test_sag_supervisor_activation] Exception:",ex)

        assert sagSupervisor != None

        acsPyClient.releaseComponent('PY_SAGSUPERVISOR')


    @pytest.mark.test_name("test_autostart_components_activation")
    def test_autostart_components_activation(self, printTestName, acsEnvironment, acsContainers, acsPyClient):

        assert acsPyClient is not None

        autoStartComponents = ["SIM_SAG_DATABASE", "PY_SAGNOTIFICATION_MANAGER", "SIM_ARRAYDATAHANDLER"]

        availableComponents = acsPyClient.availableComponents()
        for ac in availableComponents:
            if ac.name in autoStartComponents:
                assert ac.reference is not None


    @pytest.mark.test_name("test_sup_010")
    def test_sup_010(self, printTestName, acsEnvironment, acsContainers, acsPyClient, cleanSimulatedProcesses):
        """
        SUP-010 There shall be one allocated SAG pipeline for each sub-array/scheduling block operating at the same time.
        SUP-030 All SAG sub-array pipelines shall run in parallel.
        SUP-040 When a new SB is received, a dedicated SAG pipeline shall start
        in an automatic and controlled way.

        Pass criteria:
        - Check that all components are running
        - Check that the two SBs have their own SAGSubArrayManager and sub-pipeline components
        - Check that the data processing is running in parallel
        - Check that the SAGSupervisor has obtained the status of all components
        """
        assert countAliveProcessesByScriptName("sim_pipe_start.sh") == 0
        assert countAliveProcessesByScriptName("sim_pipe_data_processing.sh") == 0

        sagSimDatabase = acsPyClient.getComponent('SIM_SAG_DATABASE')
        sagSupervisor = acsPyClient.getComponent('PY_SAGSUPERVISOR')
        sleep(10)        # Give time to components to start

        outputDir = Path(sagSimDatabase.getPipelineOutputDir())
        for f in os.listdir(outputDir):
            os.remove(os.path.join(outputDir, f))

        print("New scheduling block..")
        sbId1 = "1"
        sagSupervisor.newSchedulingBlock(sbId1)
        sbId2 = "2"
        sagSupervisor.newSchedulingBlock(sbId2)

        print("Sleeping for 10 sec..")
        sleep(10)

        saId1 = sagSimDatabase.getSubArrayIDFromSB(sbId1)
        saId2 = sagSimDatabase.getSubArrayIDFromSB(sbId2)

        # Check: all components are running
        # Check: the two SBs have their own SAGSubArrayManager and sub-pipeline components
        # How: a "running" component means "active" in acs (i.e. at least one other component has a reference to it)
        # For each SB, the reference field of each corresponding components is tested to be not null. 
        activeComponents = [f"PY_SUBARRAYMANAGER_{saId1}", \
                            f"PY_SUBARRAYMANAGER_{saId2}", \
                            f"PY_RECO_MANAGERID_{saId1}", \
                            f"PY_RECO_MANAGERID_{saId2}", \
                            f"PY_DQ_MANAGERID_{saId1}", \
                            f"PY_DQ_MANAGERID_{saId2}", \
                            f"PY_SCI_MANAGERID_{saId1}", \
                            f"PY_SCI_MANAGERID_{saId2}" \
                            ]
        availableComponents = acsPyClient.availableComponents()
        for ac in availableComponents:
            # print("DEBUG: ",ac)
            if ac.name in activeComponents:
                assert ac.reference is not None


        # Check that the data processing is running in parallel
        # How: for now the data processes' loops write on a log file. First the existence of 
        # these log files is tested. Then, the processes' names are searched in the alive processes list.
        assert outputDir.joinpath(f"reco_sim_pipe_start_{sbId1}.log").exists() == True
        assert outputDir.joinpath(f"reco_sim_pipe_start_{sbId2}.log").exists() == True
        assert countAliveProcessesByScriptName("sim_pipe_start.sh") == 2


        # Check that the SAGSupervisor has obtained the status of all components
        # How: TBD


        # Start the simulation of data arrival for sub-arrays
        print("Starting the simulation of data arrival..")
        inputDataRootDir = sagSimDatabase.getSubArraysInputDataRootDir()
        simulateDataArrivalForSubArray(sbId1, inputDataRootDir, 10)
        simulateDataArrivalForSubArray(sbId2, inputDataRootDir, 10)

        print("Starting the data processing..")
        sagSupervisor.startObservation(sbId1)
        sagSupervisor.startObservation(sbId2)
        
        assert outputDir.joinpath(f"reco_sim_pipe_start_{sbId1}.log").exists() == True
        assert outputDir.joinpath(f"reco_sim_pipe_start_{sbId2}.log").exists() == True
        assert countAliveProcessesByScriptName("sim_pipe_start.sh") == 0
        assert countAliveProcessesByScriptName("sim_pipe_data_processing.sh") == 2

        print("Sleeping for 10 sec..")
        sleep(10)

        print("Stopping the processes..")
        sagSupervisor.stopSchedulingBlock(sbId1)
        sagSupervisor.stopSchedulingBlock(sbId2)

        # sleep(5)


    @pytest.mark.test_name("test_sup_040")
    def test_sup_040(self, printTestName, acsEnvironment, acsContainers, acsPyClient, cleanSimulatedProcesses):
        """
        SUP-010 There shall be one allocated SAG pipeline for each sub-array/scheduling block operating at the same time.
        SUP-030 All SAG sub-array pipelines shall run in parallel.
        SUP-040 When a new SB is received, a dedicated SAG pipeline shall start
        in an automatic and controlled way.

        Pass criteria:
        - Check that all sub-pipelines for the sub-array have stopped the data processing
        - Check the SAGSupervisor status call results in step 8
        """
        assert countAliveProcessesByScriptName("sim_pipe_start.sh") == 0
        assert countAliveProcessesByScriptName("sim_pipe_data_processing.sh") == 0

        sagSimDatabase = acsPyClient.getComponent('SIM_SAG_DATABASE')
        sagSupervisor = acsPyClient.getComponent('PY_SAGSUPERVISOR')
        sleep(10)        # Give time to components to start

        outputDir = Path(sagSimDatabase.getPipelineOutputDir())
        for f in os.listdir(outputDir):
            os.remove(os.path.join(outputDir, f))

        print("New scheduling block..")
        sbId1 = "1"
        sagSupervisor.newSchedulingBlock(sbId1)
        sbId2 = "2"
        sagSupervisor.newSchedulingBlock(sbId2)

        print("Sleeping for 10 sec..")
        sleep(10)

        saId1 = sagSimDatabase.getSubArrayIDFromSB(sbId1)
        saId2 = sagSimDatabase.getSubArrayIDFromSB(sbId2)

        # Start the simulation of data arrival for sub-arrays
        print("Starting the simulation of data arrival..")
        inputDataRootDir = sagSimDatabase.getSubArraysInputDataRootDir()
        simulateDataArrivalForSubArray(sbId1, inputDataRootDir, 10)
        simulateDataArrivalForSubArray(sbId2, inputDataRootDir, 10)

        print("Starting the data processing..")
        sagSupervisor.startObservation(sbId1)
        sagSupervisor.startObservation(sbId2)
        
        print("Sleeping for 10 sec..")
        sleep(10)

        print("Stopping the processes..")
        sagSupervisor.stopObservation(sbId1)
        sagSupervisor.stopObservation(sbId2)

        # Check that all sub-pipelines for the sub-array have stopped the data processing

        assert countAliveProcessesByScriptName("sim_pipe_data_processing.sh") == 0

        assert countAliveProcessesByScriptName("sim_pipe_start.sh") == 2

        # Check the SAGSupervisor status call results in step 8
        # TBI


    @pytest.mark.test_name("test_sup_090")
    def test_sup_090(self, printTestName, acsEnvironment, acsContainers, acsPyClient, cleanSimulatedProcesses):
        """
        SUP-110 SAG shall receive information when the execution of a Scheduling Blocks is cancelled. 
        Analysis of the data taken until the cancellation command shall be executed.

        Pass criteria:
        - Check 1: that all the pipelines for the SB are stopped
        - Check 2: that the data acquired until the SB cancellation signal has been analyzed properly
        """
        assert countAliveProcessesByScriptName("sim_pipe_start.sh") == 0
        assert countAliveProcessesByScriptName("sim_pipe_data_processing.sh") == 0

        sagSimDatabase = acsPyClient.getComponent('SIM_SAG_DATABASE')
        sagSupervisor = acsPyClient.getComponent('PY_SAGSUPERVISOR')
        sleep(10)        # Give time to components to start
        
        outputDir = Path(sagSimDatabase.getPipelineOutputDir())
        for f in os.listdir(outputDir):
            os.remove(os.path.join(outputDir, f))
        # TBI: remove input dir..
        # TBI: kill any zombie processes

        print("New scheduling block..")
        sbId1 = "1"
        sagSupervisor.newSchedulingBlock(sbId1)
        sbId2 = "2"
        sagSupervisor.newSchedulingBlock(sbId2)

        print("Sleeping for 10 sec..")
        sleep(10)

        saID1 = sagSimDatabase.getSubArrayIDFromSB(sbId1)
        saID2 = sagSimDatabase.getSubArrayIDFromSB(sbId2)



        # First test: all data files are generated, then observation is started. After x seconds 
        # the script tests if all the data files have been removed (processed).

        # Start the simulation of data arrival for sub-arrays
        print("Starting the simulation of data arrival..")
        inputDataRootDir = sagSimDatabase.getSubArraysInputDataRootDir()
      
        simThread_1 = simulateDataArrivalForSubArray(sbId1, inputDataRootDir, 10, sleepInBetween=1)
        simThread_2 = simulateDataArrivalForSubArray(sbId2, inputDataRootDir, 10, sleepInBetween=1)

        # waiting for data 
        simThread_1.join()
        simThread_2.join()

        print("Starting the data processing..")
        sagSupervisor.startObservation(sbId1)
        sagSupervisor.startObservation(sbId2)
        
        print("Sleeping for 2 sec..")
        sleep(2)

        print("Stopping the processes..")
        sagSupervisor.stopSchedulingBlock(sbId1)
        sagSupervisor.stopSchedulingBlock(sbId2)

        
        # Processes must be still alive because of check 2
        assert countAliveProcessesByScriptName("sim_pipe_data_processing.sh") == 2
        print("Waiting for files to be removed(processed)..")
        sleep(60)


        # Check 1
        assert countAliveProcessesByScriptName("sim_pipe_data_processing.sh") == 0

        # Check 2
        destDir1 = Path(inputDataRootDir).joinpath(f"datadir_{sbId1}")
        destDir2 = Path(inputDataRootDir).joinpath(f"datadir_{sbId2}")
        assert os.listdir(str(destDir1)) == []
        assert os.listdir(str(destDir2)) == []


        

    @pytest.mark.test_name("test_sup_120")
    def test_sup_120(self, printTestName, acsEnvironment, acsContainers, acsPyClient, cleanSimulatedProcesses):
        """
        SUP-140 SAG shall send monitoring information from all running sub-array pipelines and send it to 
        the ACADA Monitoring sub-system.

        Pass criteria:
        - Check that all components are running
        - Check that the simulated MON subsystem is receiving predefined monitoring information
        """
        assert countAliveProcessesByScriptName("sim_pipe_start.sh") == 0
        assert countAliveProcessesByScriptName("sim_pipe_data_processing.sh") == 0

        sagSimDatabase = acsPyClient.getComponent('SIM_SAG_DATABASE')
        sagSupervisor = acsPyClient.getComponent('PY_SAGSUPERVISOR')
        sleep(10)        # Give time to components to start


        print("New scheduling block..")
        sbId1 = "1"
        sagSupervisor.newSchedulingBlock(sbId1)
        sbId2 = "2"
        sagSupervisor.newSchedulingBlock(sbId2)

        sagMonitor = acsPyClient.getComponent('PY_SAGSUBARRAYMONITOR')

        print("Starting monitoring..")

        sagMonitor.startMonitoring()

        print("Sleeping..")

        sleep(5)

        print("Stopping monitoring..")
        sagMonitor.stopMonitoring()

        simMON = acsPyClient.getComponent('SIM_MONITORING')

        receivedMessages = simMON.getNumberOfReceivedMessages()
        print(f"MON received {receivedMessages} messages..")

        assert receivedMessages > 0 

        print("Stopping the processes..")
        sagSupervisor.stopSchedulingBlock(sbId1)        
        sagSupervisor.stopSchedulingBlock(sbId2)

    