# Copyright (C) 2020 INAF
# This software is distributed under the terms of the BSD-3-Clause license
#
# Authors:
#    Bulgarelli Andrea <andrea.bulgarelli@inaf.it>
#    Baroncelli Leonardo <leonardo.baroncelli@inaf.it>
#    Parmiggiani Nicolò <nicolo.parmiggiani@inaf.it>
#    Addis Antonio <antonio.addis@inaf.it>

from Acspy.Servants.ACSComponent import ACSComponent
from Acspy.Servants.ComponentLifecycle import ComponentLifecycle
from Acspy.Servants.ContainerServices import ContainerServices

import json
from time import sleep, time

import SAG_MODULE
import SAG_MODULE__POA

from SAG_IMPL_MODULE.SAGCallbackImpl import SAGCallbackImpl
from SAG_CUSTOM_TYPES_MODULE import SAGCallbackStatus

from UTILITY_IMPL_MODULE.ContainerUtilityImpl import ContainerUtility 

from SAG_CUSTOM_TYPES_MODULE import INIT, WORKING, DONE, ERROR

import SAGErr
import SAGErrImpl

class SAGSubArrayManagerImpl(SAG_MODULE__POA.SAGSubArrayManager, ACSComponent, ComponentLifecycle, ContainerServices):

    """This interface manages subarray configurations. It is used internally within the SAG system.

    """
    
    def __init__(self):
        ACSComponent.__init__(self)
        ContainerServices.__init__(self)

    def initialize(self):
        self._logger = self.getLogger()
        self.containerUtility = ContainerUtility()

        self.sbID = None
        self.subArrayID = None

        self.recoComponentBaseName = "SAG_PROCESS_RECO_MANAGERID_"
        self.dqComponentBaseName = "SAG_PROCESS_DQ_MANAGERID_"
        self.sciComponentBaseName = "SAG_PROCESS_SCI_MANAGERID_"

        self.sagDatabase = None
        self.SAGRecoRef = None
        self.SAGDQRef = None
        self.SAGSciRef = None



    def execute(self):
        self.sagDatabase = self.getComponent("SIM_SAG_DATABASE")

    def cleanUp(self):
        self.releaseComponent("SIM_SAG_DATABASE")

    def aboutToAbort(self):
        #Do any critical clean up
        #Continue with less critical stuff such as releasing components and other activities similar to cleanUp
        pass

    """
    Set the scheduling block ID attribute of the SubArrayManager component.
    """
    def initializeWithData(self, sbID):
        self.sbID = sbID
        subArrayID = self.sagDatabase.getSubArrayIDFromSB(sbID)
        if subArrayID == -1:
            raise SAGErrImpl.SAGSchedulingBlockIdNotFoundInDatabaseExImpl()        

        self.subArrayID = subArrayID 
        self._logger.logInfo(f"[SAGSubArrayManagerImpl {self.subArrayID}] initializeWithData() subArrayID="+str(self.subArrayID))



    """
    Activate the SAGReco, SAGSci and SAGDataQuality components for a scheduling block and start the corresponding processes.
    """
    def activateAndStartPipelines(self):

        recoComponentName = self.recoComponentBaseName + str(self.subArrayID)
        dqComponentName   = self.dqComponentBaseName   + str(self.subArrayID)
        sciComponentName  = self.sciComponentBaseName  + str(self.subArrayID)

        
        recoStatus = self.containerUtility.checkIfActive(recoComponentName)
        self._logger.logInfo(f"[SAGSubArrayManagerImpl {self.subArrayID}] ====> reco STATUS = "+str(recoStatus))
        if recoStatus == 2:
            raise SAGErrImpl.SAGRecoAlreadyActivatedExImpl()

        dqStatus = self.containerUtility.checkIfActive(dqComponentName)
        self._logger.logInfo(f"[SAGSubArrayManagerImpl {self.subArrayID}] ====> dq STATUS = "+str(dqStatus))
        if dqStatus == 2:
            raise SAGErrImpl.SAGDataQualityAlreadyActivatedExImpl()

        sciStatus = self.containerUtility.checkIfActive(sciComponentName)
        self._logger.logInfo(f"[SAGSubArrayManagerImpl {self.subArrayID}] ====> sci STATUS = "+str(sciStatus))
        if sciStatus == 2:
            raise SAGErrImpl.SAGSciAlreadyActivatedExImpl()
        

        # Due to https://ictjira.alma.cl/browse/ACS-6 in order to have asycn methods to work
        # their components must live in a different container of the caller component
        self.SAGRecoRef = self.getDynamicComponent( recoComponentName, 
                                                    "IDL:sag/SAG_MODULE/SAGReco:1.0",
                                                    "SAG_IMPL_MODULE.SAGRecoImpl",
                                                    "pyContainerForSchedulingBlock_"+self.sbID
        )
        self.SAGDQRef = self.getDynamicComponent( dqComponentName, 
                                                    "IDL:sag/SAG_MODULE/SAGDataQuality:1.0",
                                                    "SAG_IMPL_MODULE.SAGDataQualityImpl",
                                                    "pyContainerForSchedulingBlock_"+self.sbID
        )
        self.SAGSciRef = self.getDynamicComponent( sciComponentName, 
                                                    "IDL:sag/SAG_MODULE/SAGSci:1.0",
                                                    "SAG_IMPL_MODULE.SAGSciImpl",
                                                    "pyContainerForSchedulingBlock_"+self.sbID
        )


        self.SAGRecoRef.initializeWithData(self.sbID)
        self.SAGDQRef.initializeWithData(self.sbID)
        self.SAGSciRef.initializeWithData(self.sbID)

        self.SAGRecoRef.start()
        self.SAGDQRef.start()
        self.SAGSciRef.start()



    """
    [Async] Stop the reco,data quality and sci processes. Deactivate the SAGReco, SAGSci and SAGDataQuality components asynchronously.
    """   
    def stopAndDeactivatePipelines(self, sagCallback):
        
        sagCallback.working(json.dumps({}))

        recoComponentName = self.recoComponentBaseName + str(self.subArrayID)
        dqComponentName   = self.dqComponentBaseName   + str(self.subArrayID)
        sciComponentName  = self.sciComponentBaseName  + str(self.subArrayID)


        recoStatus = self.containerUtility.checkIfActive(recoComponentName)
        self._logger.logInfo(f"[SAGSubArrayManagerImpl {self.subArrayID}] ====> reco STATUS = "+str(recoStatus))
        if recoStatus == 0 or recoStatus == 1:
            sagCallback.err(json.dumps({"exception":"SAGRecoAlreadyDeactivatedExImpl"}))


        dqStatus = self.containerUtility.checkIfActive(dqComponentName)
        self._logger.logInfo(f"[SAGSubArrayManagerImpl {self.subArrayID}] ====> dq STATUS = "+str(dqStatus))
        if dqStatus == 0 or dqStatus == 1:
            sagCallback.err(json.dumps({"exception":"SAGDataQualityAlreadyDeactivatedExImpl"}))


        sciStatus = self.containerUtility.checkIfActive(sciComponentName)
        self._logger.logInfo(f"[SAGSubArrayManagerImpl {self.subArrayID}] ====> sci STATUS = "+str(sciStatus))
        if sciStatus == 0 or sciStatus == 1:
            sagCallback.err(json.dumps({"exception":"SAGSciAlreadyDeactivatedExImpl"}))

        callback = SAGCallbackImpl()

        cbObj = self.activateOffShoot(callback)

        self.SAGRecoRef.stop(cbObj)
        self.SAGDQRef.stop()
        self.SAGSciRef.stop()



        elapsed = 0
        now = time()
        while(True):

            sleep(1)

            callbackInfo = callback.getInfo()
            callbackStatus =  callbackInfo.status
            callbackDict = json.loads(callback.data)

            if callbackInfo.status == DONE:
                self._logger.logInfo(f"[SAGSubArrayManagerImpl {self.subArrayID}] stopAndDeactivatePipelines ==> callback status: {callbackStatus}, callbackdata: {callbackDict}")
                break

            elapsed += time()-now

            self._logger.logInfo(f"[SAGSubArrayManagerImpl {self.subArrayID}] stopAndDeactivatePipelines ==> waiting.. {elapsed}sec ... callback status: {callbackStatus}, callbackdata: {callbackDict}")
            
            if elapsed > 5 and not callbackDict:
                self._logger.logInfo(f"[SAGSubArrayManagerImpl {self.subArrayID}] stopAndDeactivatePipelines ==> ERROR..SAGReco callback data is empty after {elapsed}sec ... callback status: {callbackStatus}")
                sagCallback.err(json.dumps({"exception":"SAGReco component does not respond"}))
                return 

            if callbackStatus == ERROR:
                self._logger.logInfo(f"[SAGSubArrayManagerImpl {self.subArrayID}] stopAndDeactivatePipelines ==> ERROR..SAGReco callback status: {callbackStatus} => {callbackDict}")
                sagCallback.err(json.dumps(callbackDict))
                return




        self.releaseComponent(recoComponentName)
        self.releaseComponent(dqComponentName)
        self.releaseComponent(sciComponentName)

        self.SAGRecoRef = None
        self.SAGDQRef   = None
        self.SAGSciRef  = None
        
        sagCallback.done(json.dumps({}))




    """
    Start the data acquisition and processing of the subarray.
    """
    def startDataProcessing(self):
        self._logger.logInfo("[startDataProcessing]")

        recoComponentName = self.recoComponentBaseName + str(self.subArrayID)
        dqComponentName   = self.dqComponentBaseName   + str(self.subArrayID)
        sciComponentName  = self.sciComponentBaseName  + str(self.subArrayID)

        recoStatus = self.containerUtility.checkIfActive(recoComponentName)
        if recoStatus == 0 or recoStatus == 1:
            raise SAGErrImpl.SAGRecoIsNotActivatedExImpl()

        dqStatus = self.containerUtility.checkIfActive(dqComponentName)
        if dqStatus == 0 or dqStatus == 1:
            raise SAGErrImpl.SAGDataQualityIsNotActivatedExImpl()

        sciStatus = self.containerUtility.checkIfActive(sciComponentName)
        if sciStatus == 0 or sciStatus == 1:
            raise SAGErrImpl.SAGSciIsNotActivatedExImpl()

        self.SAGRecoRef.startDataProcessing()
        self.SAGDQRef.startDataProcessing()
        self.SAGSciRef.startDataProcessing()        


    """
    Stop the data acquisition and processing of the subarray.
    """
    def stopDataProcessing(self):
        self._logger.logInfo("[stopDataProcessing]")
        
        recoComponentName = self.recoComponentBaseName + str(self.subArrayID)
        dqComponentName   = self.dqComponentBaseName   + str(self.subArrayID)
        sciComponentName  = self.sciComponentBaseName  + str(self.subArrayID)

        recoStatus = self.containerUtility.checkIfActive(recoComponentName)
        if recoStatus == 0 or recoStatus == 1:
            raise SAGErrImpl.SAGRecoIsNotActivatedExImpl()

        dqStatus = self.containerUtility.checkIfActive(dqComponentName)
        if dqStatus == 0 or dqStatus == 1:
            raise SAGErrImpl.SAGDataQualityIsNotActivatedExImpl()

        sciStatus = self.containerUtility.checkIfActive(sciComponentName)
        if sciStatus == 0 or sciStatus == 1:
            raise SAGErrImpl.SAGSciIsNotActivatedExImpl()    


        self.SAGRecoRef.stopDataProcessing()
        self.SAGDQRef.stopDataProcessing()
        self.SAGSciRef.stopDataProcessing()   

    
        
    """
    [TBD] 
    """
    def updateSubArray(self, sbID):
        self._logger.logInfo("[updateSubArray] TBD")
