#!/bin/bash

config_file="config.properties"
test_config_file="test.properties"

if [ ! -f "$config_file" ]; then
  echo "$config_file doesn't exist!"
  exit 1
fi

# Append to a command a list of environment variables defined into a file
function add_properties {
  command=$1
  properties_file=$2

  while read line
  do
    if [[ ! -z "$line" ]] && [[ ! "$line" =~ ^\# ]]; then
      command="$command -D$line"
    fi
  done <<< "$(cat ../$properties_file)"

  echo $command
}

function build_core {
  cd TASMAN-bom
  mvn -q clean install
  cd ../TASMAN-core
  mvn -q clean install
  if [ "$?" -ne 0 ]; then
    echo "[ERROR] Error in ${FUNCNAME[0]}"
    exit 1
  fi
  echo "TASMAN-core built"
  cd ..
}

function test_core {
  cd TASMAN-bom
  mvn -q clean install
  cd ../TASMAN-core
  $(add_properties "mvn clean install -P test" $test_config_file)
  if [ "$?" -ne 0 ]; then
    echo "[ERROR] Error in ${FUNCNAME[0]}"
    exit 1
  fi
  cd ..
}

function build_web_glassfish {
  build_core

  # build webapp
  cd TASMAN-webapp
  $(add_properties "mvn -q clean install" $config_file)
  if [ "$?" -ne 0 ]; then
    echo "[ERROR] Error in ${FUNCNAME[0]}"
    exit 1
  fi
  echo "TASMAN-webapp built"
  cd ..
}

function build_web_tomcat {
  build_core

  # build webapp
  cd TASMAN-webapp
  $(add_properties "mvn -q clean install -P ServletContainer" $config_file)
  if [ "$?" -ne 0 ]; then
    echo "[ERROR] Error in ${FUNCNAME[0]}"
    exit 1
  fi
  echo "TASMAN-webapp built"
  cd ..
}

function build_web_embedded {
  build_core

  # build webapp
  cd TASMAN-webapp
  $(add_properties "mvn -q clean install -P ServletContainer,Jetty" $config_file)
  if [ "$?" -ne 0 ]; then
    echo "[ERROR] Error in ${FUNCNAME[0]}"
    exit 1
  fi
  echo "TASMAN-webapp built"
  cd ..

  # build embedded
  cd TASMAN-embedded
  war_file_path=`dirname ${PWD}`/TASMAN-webapp/target/tasman-webapp-*.war
  war_file_path=`ls $war_file_path`
  mvn clean -q install -Dwar_file_path=$war_file_path
  if [ "$?" -ne 0 ]; then
    echo "[ERROR] Error in ${FUNCNAME[0]}"
    exit 1
  fi
  echo "TASMAN-embedded built"
  cd ..
}

function build_installer_package {
  build_core

  # build webapp
  cd TASMAN-webapp
  mvn -q clean install -P ServletContainer,Jetty
  if [ "$?" -ne 0 ]; then
    echo "[ERROR] Error in ${FUNCNAME[0]}"
    exit 1
  fi
  echo "TASMAN-webapp built"
  cd ..

  # build embedded
  cd TASMAN-embedded
  mvn -q clean install
  if [ "$?" -ne 0 ]; then
    echo "[ERROR] Error in ${FUNCNAME[0]}"
    exit 1
  fi
  echo "TASMAN-embedded built"
  cd ..

  # copy TASMAN files
  cp TASMAN-webapp/target/tasman-webapp-*.war .
  cp TASMAN-embedded/target/tasman-embedded.jar .
  mv tasman-webapp-*.war tasman-webapp.war
  # create tar for self-extracting installer
  tar cfz tasman.tar.gz tasman-webapp.war tasman-embedded.jar
  rm tasman-webapp.war
  rm tasman-embedded.jar
  cp install_template.sh install.sh
  cat tasman.tar.gz >> install.sh
  rm tasman.tar.gz
  echo "TASMAN install.sh built"
}

function print_usage {
  echo "USAGE:"
  echo "   $0 command"
  echo " "
  echo "COMMANDS:"
  echo "   core           build only TASMAN core"
  echo "   test           run tests (using test.properties file)"
  echo "   glassfish      build GlassFish war package (using config.properties file)"
  echo "   tomcat         build Tomcat war package (using config.properties file)"
  echo "   embedded       build embedded package (using config.properties file)"
  echo "   installer      build install script (produces install.sh as output)"
  exit 0
}

if [ "$#" -ne 1 ]; then
  print_usage
fi

case "$1" in
"core")
  build_core
  ;;
"test")
  if [ ! -f "$test_config_file" ]; then
    echo "$test_config_file doesn't exist!"
    exit 1
  fi
  test_core
  ;;
"glassfish")
  build_web_glassfish
  ;;
"tomcat")
  build_web_tomcat
  ;;
"embedded")
  build_web_embedded
  ;;
"installer")
  build_installer_package
  ;;
*)
  print_usage
  ;;
esac
