/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm;

import java.util.Map;
import java.util.Objects;

/**
 * Represents a {@code KeyColumn} entity belonging to a {@link TapSchema}. Each
 * {@link Key} owns one or more {@code KeyColumn}s.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class KeyColumn extends TapSchemaEntity {

    public static final String KEY_ID_KEY = "key_id";
    public static final String FROM_COLUMN_KEY = "from_column";
    public static final String TARGET_COLUMN_KEY = "target_column";

    private static final long serialVersionUID = -3681677723432728327L;

    private Key key;

    /**
     * Only for serialization.
     */
    private KeyColumn() {
        super();
    }

    /**
     * Default constructor.
     *
     * @see TapSchemaEntity
     */
    protected KeyColumn(TapSchema tapSchema, Key key, Map<String, Object> keyColumnMetadata) {
        super(tapSchema, tapSchema.getTableModel(TapSchema.KEY_COLUMNS_TABLE), keyColumnMetadata);
        this.key = key;
    }

    /**
     * Returns the {@link Key} owning this {@code KeyColumn}.
     */
    public Key getParent() {
        return key;
    }

    /**
     * Returns the id of the parent key.
     */
    public String getKeyId() {
        return getValue(KEY_ID_KEY, String.class);
    }

    /**
     * Sets the id of the parent key.
     */
    public void setKeyId(String keyId) {
        setValue(KEY_ID_KEY, keyId);
    }

    /**
     * Returns the name of the column owning the foreign key.
     */
    public String getFromColumn() {
        return getValue(FROM_COLUMN_KEY, String.class);
    }

    /**
     * Returns the name of the column owning the primary key.
     */
    public String getTargetColumn() {
        return getValue(TARGET_COLUMN_KEY, String.class);
    }

    @Override
    public int hashCode() {
        int hash = 3;
        hash = 97 * hash + Objects.hashCode(this.key.getFromTableCompleteName());
        hash = 97 * hash + Objects.hashCode(this.getFromColumn());
        hash = 97 * hash + Objects.hashCode(this.key.getTargetTableCompleteName());
        hash = 97 * hash + Objects.hashCode(this.getTargetColumn());
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final KeyColumn other = (KeyColumn) obj;
        if (!Objects.equals(this.key.getFromTableCompleteName(), other.key.getFromTableCompleteName())) {
            return false;
        }
        if (!Objects.equals(this.getFromColumn(), other.getFromColumn())) {
            return false;
        }
        if (!Objects.equals(this.key.getTargetTableCompleteName(), other.key.getTargetTableCompleteName())) {
            return false;
        }
        if (!Objects.equals(this.getTargetColumn(), other.getTargetColumn())) {
            return false;
        }
        return true;
    }
}
