/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.webapp;

import ari.ucidy.UCDSyntax;
import ari.ucidy.UCDWord;
import ari.ucidy.UCDWordList;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;
import javax.annotation.PostConstruct;
import javax.enterprise.context.ApplicationScoped;
import javax.xml.bind.JAXB;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
@ApplicationScoped
public class WebAppConfigurationBean {

    private static final Logger LOG = LoggerFactory.getLogger(WebAppConfigurationBean.class);

    private File configFile;
    private String password;
    private Configuration configuration;

    @PostConstruct
    public void init() {
        try {
            Properties prop = new Properties();
            try (InputStream in = getClass().getClassLoader().getResourceAsStream("webapp.properties")) {
                prop.load(in);
            }
            configFile = new File(prop.getProperty("config_file_path"));
            password = prop.getProperty("password");

            if (!configFile.exists()) {

                LOG.debug("Configuration file doesn't exist: creating a new one at " + configFile.getAbsolutePath());

                configFile.getParentFile().mkdirs();
                configFile.createNewFile();
                configuration = new Configuration();
                updateConfigurationFile();

            } else {

                configuration = getOldConfigurationUpdated(configFile);

                if (configuration != null) {
                    updateConfigurationFile();
                } else {
                    // Configuration file was ok, simply unmarshal it
                    configuration = JAXB.unmarshal(configFile, Configuration.class);
                }
            }
        } catch (IOException e) {
            throw new ExceptionInInitializerError(e);
        }
    }

    public Configuration getConfig() {
        return configuration;
    }

    public void updateConfigurationFile() throws IOException {
        JAXB.marshal(configuration, configFile);
    }

    public String getPassword() {
        return password;
    }

    /**
     * Checking for old configuration and updating configuration model, if
     * necessary. This method returns null if no update is necessary.
     */
    private Configuration getOldConfigurationUpdated(File file) throws IOException {
        try {
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            DocumentBuilder db = dbf.newDocumentBuilder();
            Document doc = db.parse(file);
            String nodeName = doc.getDocumentElement().getNodeName();
            if (nodeName.equals("credentials-config")) {
                LOG.debug("Detected version 1.0.3 or lower. Extracting CredentialsConfiguration and updating the XML model.");
                CredentialsConfiguration cc = JAXB.unmarshal(file, CredentialsConfiguration.class);
                Configuration updatedConfiguration = new Configuration();
                updatedConfiguration.setCredentialsConfiguration(cc);
                return updatedConfiguration;
            }

            // Currently for version 1.0.4+ no updated are necessary.
            return null;
        } catch (Throwable t) {
            LOG.warn("Unable to parse XML configuration. Deleting it and creating a new empy one.");
            return new Configuration();
        }
    }
}
