/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.webapp;

import it.inaf.ia2.tsm.api.contract.TapSchema;
import it.inaf.ia2.tsm.api.contract.TapSchemaVersion;
import it.inaf.ia2.tsm.api.DBWrapper;
import it.inaf.ia2.tsm.api.Dao;
import it.inaf.ia2.tsm.api.DaoSchema;
import it.inaf.ia2.tsm.api.TapSchemaFactory;
import java.io.Serializable;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import javax.annotation.PostConstruct;
import javax.enterprise.context.Conversation;
import javax.enterprise.context.ConversationScoped;
import javax.faces.application.FacesMessage;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.validator.ValidatorException;
import javax.inject.Inject;
import javax.inject.Named;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
@Named("schemaSelection")
@ConversationScoped
public class SchemaSelectionBean implements Serializable {

    private static final long serialVersionUID = -5745720427701334323L;
    private static final Logger log = LoggerFactory.getLogger(SchemaSelectionBean.class);

    @Inject
    private Conversation conversation;

    @Inject
    TapSchemaEditingBean tapSchemaEditingBean;

    @Inject
    ConsistencyChecksBean consistencyChecksBean;

    private DBWrapper dbWrapper;

    private String selectedRadioOption;

    // For editing
    private List<String> allTAPSchemas;
    private String selectedTAPSchema;
    private String exposedSchemas;

    // For creation
    private String tapSchemaName;
    private List<String> allSchemas;
    private List<String> selectedSchemas;

    @PostConstruct
    public void init() {
        selectedRadioOption = "edit";
        exposedSchemas = "";
    }

    public void onPageLoad() {
        FacesContext fc = FacesContext.getCurrentInstance();
        final boolean ajaxRequest = fc.getPartialViewContext().isAjaxRequest();
        final boolean validationFailed = fc.isValidationFailed();

        if (!ajaxRequest && !validationFailed) {

            // Loading all schemas of the source database
            try {
                allSchemas = DaoSchema.getAllSchemasNames(dbWrapper.getSourceDataSource(), dbWrapper.getSourceDatabaseType());
                setSelectedSchemas(new ArrayList<String>());
            } catch (SQLException e) {
                throw new RuntimeException(e);
            }

            // Loading all schemas of the TAP_SCHEMA database
            try {
                allTAPSchemas = Dao.getAllTAPSchemasNames(dbWrapper, DaoSchema.getAllSchemasNames(dbWrapper.getTapSchemaDataSource(), dbWrapper.getTapSchemaDatabaseType()));

                if (!allTAPSchemas.isEmpty()) {
                    this.selectedTAPSchema = allTAPSchemas.get(0);
                    loadExposedSchemas();
                }
            } catch (SQLException e) {
                throw new RuntimeException(e);
            }
        }
    }

    private void loadExposedSchemas() throws SQLException {
        List<String> schemas = Dao.getExposedSchemas(dbWrapper, selectedTAPSchema);
        exposedSchemas = "";
        for (int i = 0; i < schemas.size(); i++) {
            exposedSchemas += schemas.get(i);
            if (i < schemas.size() - 1) {
                exposedSchemas += ", ";
            }
        }
    }

    public List<String> getAllTAPSchemas() {
        return allTAPSchemas;
    }

    public List<String> getAllSchemas() {
        return allSchemas;
    }

    public String getExposedSchemas() {
        return exposedSchemas;
    }

    public String getSelectedRadioOption() {
        return selectedRadioOption;
    }

    public void setSelectedRadioOption(String selectedRadioOption) {
        this.selectedRadioOption = selectedRadioOption;
    }

    public String getSelectedTAPSchema() {
        return selectedTAPSchema;
    }

    public void setSelectedTAPSchema(String selectedTAPSchema) {
        this.selectedTAPSchema = selectedTAPSchema;

        try {
            loadExposedSchemas();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    public List<String> getSelectedSchemas() {
        return selectedSchemas;
    }

    public void setSelectedSchemas(List<String> selectedSchemas) {
        this.selectedSchemas = selectedSchemas;
    }

    private String loadTapSchema(TapSchema tapSchema) {
        tapSchemaEditingBean.setTapSchema(tapSchema);
        return "tapSchemaEditing.xhtml?faces-redirect=true";
    }

    public String edit() {
        try {
            TapSchema tapSchema = TapSchemaFactory.getTapSchema(TapSchemaVersion.TAP_SCHEMA_1_IA2, dbWrapper, selectedTAPSchema, true);
            if (tapSchema.getConsistencyChecks().isInconsistent()) {
                consistencyChecksBean.setDbWrapper(dbWrapper);
                consistencyChecksBean.setTapSchema(tapSchema);
                return "consistencyChecks.xhtml?faces-redirect=true";
            } else {
                return loadTapSchema(tapSchema);
            }
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    public String create() {
        try {
            TapSchema tapSchema = TapSchemaFactory.getTapSchema(TapSchemaVersion.TAP_SCHEMA_1_IA2, dbWrapper, tapSchemaName, false);
            for (String schemaName : selectedSchemas) {
                tapSchema.addChild(schemaName);
            }
            return loadTapSchema(tapSchema);
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }
    
    public String getTapSchemaName() {
        return tapSchemaName;
    }

    public void setTapSchemaName(String tapSchemaName) {
        this.tapSchemaName = tapSchemaName;
    }

    public DBWrapper getDbWrapper() {
        return dbWrapper;
    }

    public void setDbWrapper(DBWrapper dbWrapper) {
        this.dbWrapper = dbWrapper;
    }

    public String logout() {
        conversation.end();
        return "index.xhtml?faces-redirect=true";
    }

    public void validateTapSchemaName(FacesContext context, UIComponent inputComponent, Object value) {
        String textValue = (String) value;

        String validatorMessage = null;
        if (textValue == null || textValue.isEmpty()) {
            validatorMessage = "TAP_SCHEMA name is required";
        } else if (!textValue.matches("^[a-zA-Z0-9_[-]]*$")) {
            validatorMessage = "TAP_SCHEMA name has to be a valid table name";
        } else if (allSchemas.contains(textValue)) {
            validatorMessage = "Database already contains a schema with this name. Please choose another name";
        }

        if (validatorMessage != null) {
            throw new ValidatorException(new FacesMessage(validatorMessage));
        }
    }
}
