/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.api;

import it.inaf.ia2.tsm.api.contract.Key;
import it.inaf.ia2.tsm.api.contract.KeyColumn;
import it.inaf.ia2.tsm.api.contract.TapSchema;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.regex.Pattern;

/**
 * The main implementation of {@link Key}.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class KeyImpl extends TapSchemaEntityImpl implements Key {

    private static final long serialVersionUID = -8783695875831579336L;

    private final List<KeyColumn> keyColumns;

    private String fromSchema;
    private String fromTable;
    private String targetSchema;
    private String targetTable;
    private boolean visible;

    private KeyImpl() {
        // for serialization
        super();
        keyColumns = new ArrayList<>();
    }

    public KeyImpl(DBWrapper dbWrapper, TapSchema tapSchema, String fromSchema, String fromTable, String targetSchema, String targetTable) {
        this(dbWrapper, tapSchema, fromSchema + "." + fromTable, targetSchema + "." + targetTable);
    }

    public KeyImpl(DBWrapper dbWrapper, TapSchema tapSchema, String fromTableCompleteName, String targetTableCompleteName) {
        super(dbWrapper, tapSchema);

        addProperty(FROM_TABLE_KEY, new FixedEntityProperty<>(fromTableCompleteName));
        addProperty(TARGET_TABLE_KEY, new FixedEntityProperty<>(targetTableCompleteName));

        addProperty(ID_KEY, new EditableProperty<String>());
        addProperty(UTYPE_KEY, new EditableProperty<String>());
        addProperty(DESCRIPTION_KEY, new EditableProperty<String>());
        addProperty(KEY_ID_KEY, new EditableProperty<Long>());

        keyColumns = new ArrayList<>();

        String fromTableCompleteNameSplit[] = fromTableCompleteName.split(Pattern.quote("."));
        String targetTableCompleteNameSplit[] = targetTableCompleteName.split(Pattern.quote("."));
        this.fromSchema = fromTableCompleteNameSplit[0];
        this.fromTable = fromTableCompleteNameSplit[1];
        this.targetSchema = targetTableCompleteNameSplit[0];
        this.targetTable = targetTableCompleteNameSplit[1];
        visible = false;
    }

    @Override
    public boolean isVisible() {
        return visible;
    }

    protected void setVisible(boolean visible) {
        this.visible = visible;
    }

    @Override
    public String getId() {
        return getValue(ID_KEY, String.class);
    }

    @Override
    public void setId(String id) {
        setValue(ID_KEY, id);
        for (KeyColumn keyColumn : keyColumns) {
            keyColumn.setKeyId(id);
        }
    }

    @Override
    public List<KeyColumn> getKeyColumns() {
        return Collections.unmodifiableList(keyColumns);
    }

    @Override
    public String getFromSchemaName() {
        return fromSchema;
    }

    @Override
    public String getFromTableSimpleName() {
        return fromTable;
    }

    @Override
    public String getFromTableCompleteName() {
        return getValue(FROM_TABLE_KEY, String.class);
    }

    @Override
    public String getTargetSchemaName() {
        return targetSchema;
    }

    @Override
    public String getTargetTableSimpleName() {
        return targetTable;
    }

    @Override
    public String getTargetTableCompleteName() {
        return getValue(TARGET_TABLE_KEY, String.class);
    }

    @Override
    public String getUtype() {
        return getValue(UTYPE_KEY, String.class);
    }

    @Override
    public void setUtype(String utype) {
        setValue(UTYPE_KEY, utype);
    }

    @Override
    public String getDescription() {
        return getValue(DESCRIPTION_KEY, String.class);
    }

    @Override
    public void setDescription(String description) {
        setValue(DESCRIPTION_KEY, description);
    }

    @Override
    public Long getKeyID() {
        return getValue(KEY_ID_KEY, Long.class);
    }

    @Override
    public void setKeyID(Long keyID) {
        setValue(KEY_ID_KEY, keyID);
    }

    public KeyColumn addKeyColumn(String fromColumn, String targetColumn) {
        KeyColumn keyColumn = new KeyColumnImpl(dbWrapper, tapSchema, this, fromColumn, targetColumn);
        keyColumns.add(keyColumn);
        return keyColumn;
    }

    @Override
    public void save() {
        super.save();
        for (KeyColumn keyColumn : keyColumns) {
            keyColumn.save();
        }
    }

    @Override
    public String toString() {
        return TSMUtil.printKeyInfo(this);
    }

    @Override
    public int hashCode() {
        int hash = 7;
        for (KeyColumn keyColumn : keyColumns) {
            hash = 23 * hash + Objects.hashCode(keyColumn);
        }
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }

        final KeyImpl other = (KeyImpl) obj;
        List<KeyColumn> otherKeyColumns = other.getKeyColumns();

        // Comparing each key column
        if (keyColumns.size() != otherKeyColumns.size()) {
            return false;
        }
        for (int i = 0; i < keyColumns.size(); i++) {
            if (!keyColumns.get(i).equals(otherKeyColumns.get(i))) {
                return false;
            }
        }

        return true;
    }
}
