/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.api;

import java.io.Serializable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Represent an {@code EntityProperty} which value can be modified by the user.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class EditableProperty<T> implements EntityProperty, Serializable {

    private static final long serialVersionUID = 4896192667771442640L;
    private static final Logger log = LoggerFactory.getLogger(EditableProperty.class);

    private final T defaultValue;
    private final boolean allowsNull;
    private T originalValue;
    private T value;

    public EditableProperty() {
        this.defaultValue = null;
        this.allowsNull = true;
        this.init(defaultValue);
    }

    public EditableProperty(T defaultValue, boolean allowsNull) {
        this.defaultValue = defaultValue;
        this.allowsNull = allowsNull;
        this.init(defaultValue);
    }

    @Override
    public <X> X getValue(Class<X> type) {
        return (X) value;
    }

    public <X> X getOriginalValue(Class<X> type) {
        return (X) originalValue;
    }

    public <X> void setValue(X value) {
        if (value == null && !allowsNull) {
            throw new IllegalArgumentException("This EditableProperty instance doesn't allow null values");
        }
        this.value = (T) value;
    }

    @Override
    public boolean isChanged() {
        if (originalValue == null) {
            return value != null;
        }
        return !originalValue.equals(value);
    }

    @Override
    public final <X> void init(X initialValue) {
        setValue(initialValue);
        this.originalValue = (T) initialValue;
    }

    public void save() {
        this.originalValue = value;
    }
}
