/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.api;

import it.inaf.ia2.tsm.api.contract.DatabaseType;
import java.io.Serializable;
import java.sql.Connection;
import java.sql.SQLException;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is used to silently manage the possibility to have separate data
 * sources for the TAP_SCHEMA schema and it source schema (the schema from which
 * it takes the information).<br>
 * An API user asks, for example, {@link getSourceDataSource()}, and the
 * <code>DBWrapper</code> returns the correct <code>DataSource</code>, both if
 * it is separated from the TAP_SCHEMA <code>DataSource</code> or if they are
 * the same object.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class DBWrapper implements Serializable {

    private static final long serialVersionUID = 1721030677924066695L;
    private final static Logger log = LoggerFactory.getLogger(DBWrapper.class);

    // Same credentials
    private Credentials credentials;

    // Separated credentials
    private Credentials sourceCredentials;
    private Credentials tapSchemaCredentials;

    private final DataSourcesWrapper dataSources;

    private DBWrapper() {
        dataSources = new DataSourcesWrapper();
    }

    /**
     * Constructor to use if the source schema credentials and the TAP_SCHEMA
     * credentials are the same.
     */
    public DBWrapper(Credentials credentials) {
        this();
        this.credentials = credentials;
    }

    /**
     * Constructor to use if the source schema credentials are different from
     * the TAP_SCHEMA credentials.
     */
    public DBWrapper(Credentials sourceCredentials, Credentials tapSchemaCredentials) {
        this();
        this.sourceCredentials = sourceCredentials;
        this.tapSchemaCredentials = tapSchemaCredentials;
    }

    public DataSource getSourceDataSource() {
        return dataSources.getSourceDataSource();
    }

    public DataSource getTapSchemaDataSource() {
        return dataSources.getTapSchemaDataSource();
    }

    public Connection getSourceConnection() throws SQLException {
        return getSourceDataSource().getConnection();
    }

    public Connection getTapSchemaConnection() throws SQLException {
        return getTapSchemaDataSource().getConnection();
    }

    public Credentials getSourceCredentials() {
        if (credentials != null) {
            return credentials;
        }
        return sourceCredentials;
    }

    public Credentials getTapSchemaCredentials() {
        if (credentials != null) {
            return credentials;
        }
        return tapSchemaCredentials;
    }

    public DatabaseType getSourceDatabaseType() {
        return getSourceCredentials().getDatabaseType();
    }

    public DatabaseType getTapSchemaDatabaseType() {
        return getTapSchemaCredentials().getDatabaseType();
    }

    /**
     * Test both the connection to the TAP_SCHEMA <code>DataSource</code> and
     * its source <code>DataSource</code>.
     *
     * @throws SQLException if it is not possible to connect to the
     * <code>DataSource</code>.
     */
    public void testConnections() throws SQLException {
        Connection connection;
        if (credentials != null) {
            connection = dataSources.getSourceDataSource().getConnection();
            connection.close();
        } else {
            connection = dataSources.getSourceDataSource().getConnection();
            connection.close();
            connection = dataSources.getTapSchemaDataSource().getConnection();
            connection.close();
        }
    }

    /**
     * @return true if the TAP_SCHEMA <code>DataSource</code> is different from
     * its source <code>DataSource</code>, false otherwise.
     */
    public boolean isSeparatedSources() {
        return dataSources.isSeparatedSources();
    }

    private class DataSourcesWrapper implements Serializable {

        private static final long serialVersionUID = -7025255003212206748L;

        private transient DataSource dataSource;
        private transient DataSource sourceDataSource;
        private transient DataSource tapSchemaDataSource;

        public boolean isSeparatedSources() {
            return dataSource == null;
        }

        public DataSource getTapSchemaDataSource() {
            if (credentials != null) {
                if (dataSource == null) {
                    dataSource = TSMUtil.createDataSource(credentials);
                }
                return dataSource;
            }
            if (tapSchemaDataSource == null) {
                tapSchemaDataSource = TSMUtil.createDataSource(tapSchemaCredentials);
            }
            return tapSchemaDataSource;
        }

        public DataSource getSourceDataSource() {
            if (credentials != null) {
                if (dataSource == null) {
                    dataSource = TSMUtil.createDataSource(credentials);
                }
                return dataSource;
            }
            if (sourceDataSource == null) {
                sourceDataSource = TSMUtil.createDataSource(sourceCredentials);
            }
            return sourceDataSource;
        }
    }
}
