/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.webapp;

import it.inaf.ia2.tsm.api.contract.ChildEntity;
import it.inaf.ia2.tsm.api.contract.Column;
import it.inaf.ia2.tsm.api.contract.EntitiesContainer;
import it.inaf.ia2.tsm.api.contract.Key;
import it.inaf.ia2.tsm.api.contract.KeyColumn;
import it.inaf.ia2.tsm.api.contract.Schema;
import it.inaf.ia2.tsm.api.contract.Status;
import it.inaf.ia2.tsm.api.contract.Table;
import it.inaf.ia2.tsm.api.contract.TapSchema;
import it.inaf.ia2.tsm.api.contract.TapSchemaEntity;
import it.inaf.ia2.tsm.api.UpdateOperations;
import java.io.Serializable;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import javax.enterprise.context.Conversation;
import javax.enterprise.context.ConversationScoped;
import javax.faces.context.FacesContext;
import javax.inject.Inject;
import javax.inject.Named;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
@Named("tapSchemaEditing")
@ConversationScoped
public class TapSchemaEditingBean implements Serializable {

    private static final long serialVersionUID = -6251004452688984277L;
    private static final Logger log = LoggerFactory.getLogger(TapSchemaEditingBean.class);

    private static final String COLUMNS_COMPONENT_ID = "main:columns-list";

    @Inject
    private Conversation conversation;

    @Inject
    SchemaSelectionBean schemaSelection;

    private TapSchema tapSchema;
    private Schema selectedSchema;
    private Table selectedTable;
    private Column selectedColumn;
    private UpdateOperations updateOperations;

    private EntitiesContainer currentAddingContainer;
    private List<AddableItem> currentAddables;

    @Inject
    private SearchUCDDialog searchUCDDialog;

    public Schema getSelectedSchema() {
        return selectedSchema;
    }

    public Table getSelectedTable() {
        return selectedTable;
    }

    public Column getSelectedColumn() {
        return selectedColumn;
    }

    public void setSelectedSchema(Schema selectedSchema) {
        this.selectedSchema = selectedSchema;
        if (selectedSchema == null) {
            setSelectedTable(null);
        } else {
            List<Table> tables = selectedSchema.getAddedOrRemovedChildren();
            if (tables.isEmpty()) {
                setSelectedTable(null);
            } else {
                setSelectedTable(tables.get(0));
            }
        }
    }

    public void setSelectedTable(Table selectedTable) {
        this.selectedTable = selectedTable;
        if (selectedTable == null) {
            setSelectedColumn(null);
        } else {
            List<Column> columns = selectedTable.getAddedOrRemovedChildren();
            if (columns.isEmpty()) {
                setSelectedColumn(null);
            } else {
                setSelectedColumn(columns.get(0));
            }
        }
    }

    public void setSelectedColumn(Column selectedColumn) {
        this.selectedColumn = selectedColumn;
    }

    public UpdateOperations getUpdateOperations() {
        return updateOperations;
    }

    public String getUpdatedValues(TapSchemaEntity entity) {
        StringBuilder sb = new StringBuilder();
        boolean first = true;
        for (String key : entity.getPropertiesKeys()) {
            if (entity.isChanged(key)) {
                if (!first) {
                    sb.append(", ");
                }
                sb.append(key);
            }
        }
        return sb.toString();
    }

    public static class AddableItem implements Serializable {

        private static final long serialVersionUID = 2732253307571391962L;

        private final String name;
        private boolean selected;

        AddableItem(String name) {
            this.name = name;
        }

        public String getName() {
            return name;
        }

        public boolean isSelected() {
            return selected;
        }

        public void setSelected(boolean selected) {
            this.selected = selected;
        }
    }

    public boolean toRemove(ChildEntity child) {
        return child.getStatus() == Status.REMOVED_NOT_PERSISTED || child.getStatus() == Status.TO_REMOVE;
    }

    public String getForeignKeyString(Column column) throws SQLException {

        Key foreignKey = column.getForeignKey();

        if (foreignKey == null) {
            return null;
        }

        if (foreignKey.isVisible()) {

            String columnName = column.getName();

            StringBuilder sb = new StringBuilder();

            sb.append(foreignKey.getFromTableCompleteName());
            sb.append(".");
            sb.append(columnName);
            sb.append(" -> ");
            for (KeyColumn keyColumn : foreignKey.getKeyColumns()) {
                if (keyColumn.getFromColumn().equals(columnName)) {
                    sb.append(foreignKey.getTargetTableCompleteName());
                    sb.append(keyColumn.getTargetColumn());
                    break;
                }
            }

            return sb.toString();
        }

        return null;
    }

    public TapSchema getTapSchema() {
        return tapSchema;
    }

    private void setupSelected() {
        selectedSchema = null;
        selectedTable = null;
        selectedColumn = null;
        List<Schema> schemas = tapSchema.getAddedOrRemovedChildren();
        if (!schemas.isEmpty()) {
            selectedSchema = schemas.get(0);
            List<Table> tables = selectedSchema.getAddedOrRemovedChildren();
            if (!tables.isEmpty()) {
                selectedTable = tables.get(0);
                List<Column> columns = selectedTable.getAddedOrRemovedChildren();
                if (!columns.isEmpty()) {
                    selectedColumn = columns.get(0);
                }
            }
        }
    }

    public void setTapSchema(TapSchema tapSchema) {
        this.tapSchema = tapSchema;
        setupSelected();
    }

    public void update() throws SQLException {
        tapSchema.save();
        setupSelected();
    }

    public SearchUCDDialog getSearchUCDDialog() {
        return searchUCDDialog;
    }

    public String back() {
        return "schemaSelection.xhtml?faces-redirect=true";
    }

    public String logout() {
        conversation.end();
        return "index.xhtml?faces-redirect=true";
    }

    public void undoRemove(ChildEntity entity) throws SQLException {
        // re-add what was removed
        entity.getParent().addChild(entity.getName());
    }

    public EntitiesContainer getCurrentAddingContainer() {
        return currentAddingContainer;
    }

    public void openAddablesModal(EntitiesContainer<?> currentAddingContainer) {
        this.currentAddingContainer = currentAddingContainer;
        this.currentAddables = new ArrayList<>();
        for (String name : currentAddingContainer.getAddableChildrenNames()) {
            if (currentAddingContainer instanceof TapSchema && !tapSchema.exists() && name.equals(tapSchema.getName())) {
                // we can't add the TAP_SCHEMA into itself when it doesn't
                // created yet.
                continue;
            }
            currentAddables.add(new AddableItem(name));
        }
    }

    public void checkAllEntities(boolean value) {
        for (AddableItem item : currentAddables) {
            item.setSelected(value);
        }
    }

    public void addSelected() throws SQLException {
        TapSchemaEntity lastAddedEntity = null;
        for (AddableItem item : currentAddables) {
            if (item.isSelected()) {
                lastAddedEntity = currentAddingContainer.addChild(item.getName());
            }
        }
        if (lastAddedEntity != null) {
            if (lastAddedEntity instanceof Schema) {
                setSelectedSchema((Schema) lastAddedEntity);
            } else if (lastAddedEntity instanceof Table) {
                setSelectedTable((Table) lastAddedEntity);
            } else if (lastAddedEntity instanceof Column) {
                setSelectedColumn((Column) lastAddedEntity);
            }
        }
    }

    public List<AddableItem> getCurrentAddables() {
        return currentAddables;
    }

    public void saveUCD() {
        if (!FacesContext.getCurrentInstance().isValidationFailed()) {

            if (searchUCDDialog.isManualInsertion()) {
                selectedColumn.setUCD(searchUCDDialog.getUCDManualText());
            } else {
                selectedColumn.setUCD(searchUCDDialog.getSelectedUCD());
            }

            // New UCD is set and we can notify the client to close the UCD Search modal dialog.
            CustomPartialResponseWriter.getCurrentInstance().addCustomJSUpdate(new JSUpdateHandler() {

                @Override
                public String getUpdate() {
                    return "true";
                }
            });
        }
    }

    public void displayUpdateOperations() {
        updateOperations = new UpdateOperations(tapSchema);
    }

    public void openUCDDialog() throws Exception {
        searchUCDDialog.setDefault();
        String description = selectedColumn.getDescription();
        if (description != null && !description.isEmpty()) {
            searchUCDDialog.setDescription(description);
            searchUCDDialog.search(description);
        }
    }

    public void textInputChanged(TapSchemaEntity entity, String key) {
        final boolean isChanged = entity.isChanged(key);
        CustomPartialResponseWriter.getCurrentInstance().addCustomJSUpdate(new JSUpdateHandler() {

            @Override
            public String getUpdate() {
                return isChanged + "";
            }
        });
    }

    public void removeColumn(String name) {
        selectedTable.removeChild(name);

        Integer selectedColumnIndex = null;
        if (selectedColumn != null) {
            int index = 0;
            for (Column column : selectedTable.getAddedOrRemovedChildren()) {
                if (column.getName().equals(selectedColumn.getName())) {
                    selectedColumnIndex = index;
                    break;
                }
                index++;
            }
        }

        CustomPartialResponseWriter writer = CustomPartialResponseWriter.getCurrentInstance();

        final String updateResult = selectedColumnIndex == null ? "{}" : "{\"selectedColumn\":" + selectedColumnIndex + "}";

        writer.encodeComponent(COLUMNS_COMPONENT_ID);
        writer.addCustomJSUpdate(COLUMNS_COMPONENT_ID, new JSUpdateHandler() {

            @Override
            public String getUpdate() {
                return updateResult;
            }
        });
    }

    public void undoRemoveColumn() throws SQLException {
        this.undoRemove(selectedColumn);

        int i = 0;
        for (Column column : selectedTable.getAddedOrRemovedChildren()) {
            if (column.getName().equals(selectedColumn.getName())) {
                break;
            }
            i++;
        }

        final int columnIndex = i;

        CustomPartialResponseWriter writer = CustomPartialResponseWriter.getCurrentInstance();
        writer.addCustomJSUpdate(COLUMNS_COMPONENT_ID, new JSUpdateHandler() {

            @Override
            public String getUpdate() {
                return columnIndex + "";
            }
        });
    }

    public String reload() {
        return schemaSelection.edit();
    }
}
