/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.api.contract;

import java.sql.SQLException;
import java.util.List;

/**
 * Objects of this class contain zero or more {@link ChildEntity} objects.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public interface EntitiesContainer<T extends ChildEntity> {

    /**
     * Returns a specific child of the container, selecting it by name and
     * filtering it by a set of possible status: if the child doesn't have one
     * of the status specified, this method returns null.<br>
     * <strong>If no status is specified it is assumed that all status are
     * valid.</strong>
     */
    T getChild(String childName, Status... status);

    /**
     * Add a {@link ChildEntity} specifying its name. If the child entity has
     * never been read from the database it is loaded when this method is
     * called. If the entity was marked as removed, it will marked as added
     * again.
     */
    T addChild(String childName) throws SQLException;

    /**
     * Remove a {@link ChildEntity} specifying its name.<br>
     * The child isn't really removed but only marked as removed (the removal
     * will happen when the {@link TapSchema#save()} method will be called).
     */
    void removeChild(String childName);

    /**
     * Retrieve a list of names of child entities that can be added.<br> The
     * list contains only names of entities that have never been added: entities
     * which instance don't exist yet or entities having
     * {@link Status} {@code LOADED}.
     */
    List<String> getAddableChildrenNames();

    /**
     * Retrieve a list of children filtering them by a set of possible
     * status.<br>
     * <strong>If no status is specified it is assumed that all status are
     * valid.</strong>
     */
    List<T> getChildren(Status... statuses);

    /**
     * Retrieve all children having {@link Status} {@code ADDED_PERSISTED} or
     * {@code ADDED_NOT_PERSISTED}.
     */
    List<T> getAddedChildren();

    /**
     * Retrieve all children having {@link Status} {@code ADDED_PERSISTED},
     * {@code ADDED_NOT_PERSISTED}, {@code TO_REMOVE} or
     * {@code REMOVED_NOT_PERSISTED}.
     */
    List<T> getAddedOrRemovedChildren();
}
