/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.api;

import it.inaf.ia2.tsm.api.contract.Column;
import it.inaf.ia2.tsm.api.contract.Key;
import it.inaf.ia2.tsm.api.contract.KeyColumn;
import it.inaf.ia2.tsm.api.contract.Schema;
import it.inaf.ia2.tsm.api.contract.Status;
import it.inaf.ia2.tsm.api.contract.Table;
import it.inaf.ia2.tsm.api.contract.TapSchema;
import it.inaf.ia2.tsm.api.contract.TapSchemaVersion;
import java.io.Serializable;
import java.sql.SQLException;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The main implementation of {@link TapSchema}.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class TapSchemaImpl implements TapSchema, Serializable {

    private static final long serialVersionUID = 1678083091602571256L;
    private static final Logger log = LoggerFactory.getLogger(TapSchemaImpl.class);

    private final Map<String, Schema> schemas;
    private final Set<Key> allKeys;
    private final ConsistencyChecks consistencyChecks;

    private TapSchemaVersion version;
    private DBWrapper dbWrapper;
    private String tapSchemaName;
    private boolean exists;

    private TapSchemaImpl() {
        // for serialization
        schemas = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);
        allKeys = new HashSet<>();
        consistencyChecks = new ConsistencyChecks();
    }

    protected TapSchemaImpl(TapSchemaVersion version, DBWrapper dbWrapper, String tapSchemaName, boolean exists) throws SQLException {
        this.version = version;
        this.dbWrapper = dbWrapper;
        this.tapSchemaName = tapSchemaName;
        this.exists = exists;

        schemas = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);
        allKeys = new HashSet<>();
        consistencyChecks = new ConsistencyChecks();

        // Initializing schemas map
        for (String schemaName : DaoSchema.getAllSchemasNames(dbWrapper.getSourceDataSource(), dbWrapper.getSourceDatabaseType())) {
            schemas.put(schemaName, null);
        }
        schemas.put(tapSchemaName, null); // the TAP_SCHEMA contains itself

        if (exists) {
            DaoSchema.fillSavedSchemas(dbWrapper, (this));
            DaoTable.fillSavedTables(dbWrapper, (this));
            DaoColumn.fillSavedColumns(dbWrapper, (this));
            DaoKey.fillSavedKeys(dbWrapper, (this));
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getName() {
        return tapSchemaName;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public TapSchemaVersion getVersion() {
        return version;
    }

    private void loadSchemaKeys(String schemaName) throws SQLException {
        for (Key key : DaoKey.getSchemaKeys(dbWrapper, this, schemaName)) {
            if (!allKeys.contains(key)) {
                allKeys.add(key);
            }
        }
    }

    protected Set<Key> getAllKeys() {
        return allKeys;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Schema addChild(String schemaName) throws SQLException {
        log.debug("Adding schema {}", schemaName);

        Schema schema;
        
        if (!schemas.containsKey(schemaName)) {
            
            consistencyChecks.addUnexistingSchema(schemaName);
            schema = null;
        } else {

            schema = schemas.get(schemaName);

            if (schema == null) {
                schema = new SchemaImpl(dbWrapper, this, schemaName);
                schema.setStatus(Status.ADDED_NOT_PERSISTED);
                schemas.put(schemaName, schema);
                loadSchemaKeys(schemaName);
            } else {
                switch (schema.getStatus()) {
                    case TO_REMOVE:
                        schema.setStatus(Status.ADDED_PERSISTED);
                        break;
                    case REMOVED_NOT_PERSISTED:
                        schema.setStatus(Status.ADDED_NOT_PERSISTED);
                        break;
                    default:
                        throw new IllegalArgumentException("Cannot add the schema " + schemaName + ". Invalid status. Schema status is " + schema.getStatus());
                }
            }
        }

        checkKeys();

        return schema;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void removeChild(String schemaName) {
        log.debug("Removing schema {}", schemaName);

        if (!schemas.containsKey(schemaName)) {
            throw new IllegalArgumentException("The database doesn't contains a schema named " + schemaName);
        }

        Schema schema = schemas.get(schemaName);
        if (schema == null || schema.getStatus() == Status.LOADED) {
            throw new IllegalArgumentException("Cannot remove the schema " + schemaName + ". It has never been added.");
        }

        if (schema.getStatus() == Status.ADDED_NOT_PERSISTED) {
            schema.setStatus(Status.REMOVED_NOT_PERSISTED);
        } else if (schema.getStatus() == Status.ADDED_PERSISTED) {
            schema.setStatus(Status.TO_REMOVE);
        }

        checkKeys();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Schema getChild(String childName, Status... statuses) {
        return TSMUtil.getChild(schemas, childName, statuses);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Schema> getChildren(Status... statuses) {
        return TSMUtil.getChildrenByStatus(schemas.values(), statuses);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<String> getAddableChildrenNames() {
        return TSMUtil.getAddableChildrenNames(schemas);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Schema> getAddedChildren() {
        return getChildren(Status.ADDED_PERSISTED, Status.ADDED_NOT_PERSISTED);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Schema> getAddedOrRemovedChildren() {
        return getChildren(Status.ADDED_PERSISTED, Status.ADDED_NOT_PERSISTED, Status.TO_REMOVE, Status.REMOVED_NOT_PERSISTED);
    }

    /**
     * This method has to be used after TAP_SCHEMA modifications are committed
     * to the database, in order to remove from the memory the schemas with
     * status {@code Status.TO_REMOVE} or {@code Status.REMOVED_NOT_PERSISTED}.
     */
    protected void cleanSchema(String schemaName) {
        if (!schemas.containsKey(schemaName)) {
            throw new IllegalArgumentException("The TAP_SCHEMA doesn't contain the schema " + schemaName);
        }
        schemas.put(schemaName, null);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void save() throws SQLException {

        Dao.save(dbWrapper, this);

        if (!exists) {
            // Adding TAP_SCHEMA into TAP_SCHEMA
            Schema tapSchemaSchema = addChild(tapSchemaName);
            for (String tableName : tapSchemaSchema.getAddableChildrenNames()) {
                Table table = tapSchemaSchema.addChild(tableName);
                for (String columnName : table.getAddableChildrenNames()) {
                    table.addChild(columnName);
                }
            }
            log.debug(this.toString());
            TSMUtil.putInfoIntoTapSchemaSchema(tapSchemaSchema);

            exists = true; // important!

            Dao.save(dbWrapper, this); // save again
        }

        exists = true;

        consistencyChecks.getInconsistencies().clear();
    }

    /**
     * Retrieve the maximum key id from all the schemas that are added into the
     * TAP_SCHEMA.
     *
     * @return the maximum key, if it exists, zero otherwise.
     */
    public int getMaxKeyId() {
        int maxKeyId = 0;
        for (Key key : allKeys) {
            if (key.getId() != null) {
                int keyId = Integer.parseInt(key.getId());
                if (keyId > maxKeyId) {
                    maxKeyId = keyId;
                }
            }
        }
        return maxKeyId;
    }

    /**
     * Set keys visibility based on other entities visibility (a key is visible
     * if all schemas, tables and columns involved have
     * {@link Status} {@code ADDED_PERSISTED} or {@code ADDED_NOT_PERSISTED}).
     */
    protected void checkKeys() {

        int currentKey = getMaxKeyId() + 1;

        for (Key key : allKeys) {

            ((KeyImpl) key).setVisible(false);

            Schema fromSchema = getChild(key.getFromSchemaName(), Status.ADDED_PERSISTED, Status.ADDED_NOT_PERSISTED);
            Schema targetSchema = getChild(key.getTargetSchemaName(), Status.ADDED_PERSISTED, Status.ADDED_NOT_PERSISTED);
            if (fromSchema != null && targetSchema != null) {

                Table fromTable = fromSchema.getChild(key.getFromTableSimpleName(), Status.ADDED_PERSISTED, Status.ADDED_NOT_PERSISTED);
                Table targetTable = targetSchema.getChild(key.getTargetTableSimpleName(), Status.ADDED_PERSISTED, Status.ADDED_NOT_PERSISTED);

                if (fromTable != null && targetTable != null) {

                    boolean allColumnsVisible = true;

                    for (KeyColumn keyColumn : key.getKeyColumns()) {

                        Column fromColumn = fromTable.getChild(keyColumn.getFromColumn(), Status.ADDED_PERSISTED, Status.ADDED_NOT_PERSISTED);
                        Column targetColumn = targetTable.getChild(keyColumn.getTargetColumn(), Status.ADDED_PERSISTED, Status.ADDED_NOT_PERSISTED);

                        if (fromColumn == null || targetColumn == null) {
                            allColumnsVisible = false;
                            break;
                        }
                    }

                    if (allColumnsVisible) {
                        ((KeyImpl) key).setVisible(true);
                        if (key.getId() == null) {
                            key.setId(currentKey + "");
                            currentKey++;
                        }
                    }
                }
            }
        }
//        for (Key key : allKeys) {
//            log.debug("{} [{}]", key, key.getStatus());
//        }
    }

    public void addFictitiousKey(Table fromTable, String[] fromColumns, Table targetTable, String[] targetColumns) {
        KeyImpl key = new KeyImpl(dbWrapper, this, fromTable.getCompleteName(), targetTable.getCompleteName());
        key.setId((getMaxKeyId() + 1) + "");

        for (int i = 0; i < fromColumns.length; i++) {
            key.addKeyColumn(fromColumns[i], targetColumns[i]);
        }

        fromTable.addFromKey(key);
        targetTable.addTargetKey(key);

        allKeys.add(key);
        checkKeys();
    }

    /**
     * Print all TAP_SCHEMA tree (useful for debugging).
     */
    @Override
    public String toString() {

        StringBuilder sb = new StringBuilder("\n");

        sb.append(String.format(">> TAP_SCHEMA %s <<\n", tapSchemaName));

        for (Schema schema : getChildren()) {
            sb.append("--");
            sb.append(schema.getName());
            sb.append(String.format(" [%s]", schema.getStatus()));
            sb.append("\n");
            List<Table> tables = schema.getChildren();
            for (int i = 0; i < tables.size(); i++) {
                Table table = tables.get(i);
                sb.append("   |--");
                sb.append(table.getName());
                sb.append(String.format(" [%s]", table.getStatus()));
                sb.append("\n");

                String padder = i < tables.size() - 1 ? "   |   " : "       ";

                for (Column column : table.getChildren()) {
                    sb.append(padder);
                    sb.append("|--");
                    sb.append(column.getName());
                    sb.append(String.format(" [%s]", column.getStatus()));
                    sb.append("\n");
                }

                if (table.getAllFromKeys().size() > 0) {
                    sb.append(padder);
                    sb.append("** From keys **\n");
                    for (Key fromKey : table.getAllFromKeys()) {
                        sb.append(padder);
                        sb.append("* ");
                        sb.append(fromKey.toString());
                        sb.append(String.format(" [visible=%s]", fromKey.isVisible()));
                        sb.append("\n");
                    }
                }
                if (table.getAllTargetKeys().size() > 0) {
                    sb.append(padder);
                    sb.append("** Target keys **\n");
                    for (Key targetKey : table.getAllTargetKeys()) {
                        sb.append(padder);
                        sb.append("* ");
                        sb.append(targetKey.toString());
                        sb.append(String.format(" [visible=%s]", targetKey.isVisible()));
                        sb.append("\n");
                    }
                }

                sb.append("\n");
            }
        }

        return sb.toString();
    }

    @Override
    public boolean exists() {
        return exists;
    }

    @Override
    public ConsistencyChecks getConsistencyChecks() {
        return consistencyChecks;
    }
}
