/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.api;

import it.inaf.ia2.tsm.api.contract.DatabaseType;
import it.inaf.ia2.tsm.api.contract.Schema;
import it.inaf.ia2.tsm.api.contract.Status;
import it.inaf.ia2.tsm.api.contract.TapSchema;
import it.inaf.ia2.tsm.api.contract.TapSchemaEntity;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Utility class that contains static methods for managing {@link Schema}s into
 * the database.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class DaoSchema {

    private static final Logger log = LoggerFactory.getLogger(DaoSchema.class);

    /**
     * Retrieve the list of the names of the all the schemas contained into the
     * database specified by the <code>DataSource</code> parameter.
     *
     * @return list of schemas names alphabetically and case insensitively
     * ordered.
     */
    public static List<String> getAllSchemasNames(DataSource dataSource, DatabaseType dbType) throws SQLException {

        log.debug("getAllSchemasNames");

        String query;
        if (dbType == DatabaseType.MYSQL) {
            query = "SHOW DATABASES";
        } else if (dbType == DatabaseType.POSTGRES) {
            query = "SELECT schema_name FROM information_schema.schemata";
        } else {
            throw new UnsupportedOperationException("Database type " + dbType + " not supported");
        }

        log.debug("Executing query {}", query);

        List<String> allSchemas = new ArrayList<>();

        try (Connection connection = dataSource.getConnection();
                Statement statement = connection.createStatement();
                ResultSet resultSet = statement.executeQuery(query)) {
            while (resultSet.next()) {
                allSchemas.add(resultSet.getString(1));
            }
        }

        log.debug("{} schemas found", allSchemas.size());

        return TSMUtil.sortStringsList(allSchemas);
    }

    /**
     * Retrieves saved {@code Schema}s from the database and add them into the
     * specified {@code TapSchema}.
     */
    protected static void fillSavedSchemas(DBWrapper dbWrapper, final TapSchema tapSchema) throws SQLException {

        log.debug("fillSavedSchemas");

        SelectQueryBuilder selectQueryBuilder = new SelectQueryBuilder(dbWrapper.getTapSchemaDatabaseType(), tapSchema, TapSchema.SCHEMAS_TABLE) {

            @Override
            protected TapSchemaEntity getEntity(ResultSet rs) throws SQLException {
                String schemaName = rs.getString("schema_name");
                Schema schema = tapSchema.addChild(schemaName);
                if(schema == null) {
                    return null;
                }
                schema.setStatus(Status.ADDED_PERSISTED);
                return schema;
            }
        };

        selectQueryBuilder.executeQuery(dbWrapper.getTapSchemaConnection());
    }

    /**
     * Save a new {@code Schema} into the TAP_SCHEMA schema.
     */
    protected static void insertNewSchema(DatabaseType dbType, Connection connection, TapSchema tapSchema, Schema schema) throws SQLException {

        log.debug("insertNewSchema");

        InsertQueryBuilder insertQueryBuilder = new InsertQueryBuilder(dbType, tapSchema, schema, TapSchema.SCHEMAS_TABLE);
        insertQueryBuilder.executeQuery(connection);
    }

    /**
     * Updates an existing {@code Schema}.
     */
    protected static void updateSchema(DatabaseType dbType, Connection connection, TapSchema tapSchema, Schema schema) throws SQLException {

        UpdateQueryBuilder updateQueryBuilder = new UpdateQueryBuilder(dbType, tapSchema, schema, TapSchema.SCHEMAS_TABLE, "schema_name = ?");

        String query = updateQueryBuilder.getQuery();

        try (PreparedStatement statement = connection.prepareStatement(query)) {

            log.debug("Executing query {}", query);

            int i = updateQueryBuilder.addStatementValues(statement);
            statement.setString(i, schema.getName());

            statement.executeUpdate();
        }
    }
}
