/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.oats.ia2.tapschemamanager.webapp;

import ari.ucd.UCD;
import ari.ucd.UCDParser;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Serializable wrapper for the ari.ucd.UCD class. Contains only a subset of the
 * original properties.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class ParsedUCD implements Serializable {

    private static final long serialVersionUID = 527435700858258310L;

    private final boolean valid;
    private final boolean recognised;
    private final boolean recommended;
    private final List<String> errors;
    private final List<String> advices;
    private final String suggestedUCD;

    public ParsedUCD(UCD parsedUCD) {
        valid = parsedUCD.isFullyValid();
        recognised = parsedUCD.isAllRecognised();
        recommended = parsedUCD.isAllRecommended();

        errors = new ArrayList<>();
        Iterator<String> errorIte = parsedUCD.getErrors();
        while (errorIte.hasNext()) {
            errors.add(errorIte.next());
        }

        advices = new ArrayList<>();
        Iterator<String> adviceIte = parsedUCD.getAdvice();
        while (adviceIte.hasNext()) {
            advices.add(adviceIte.next());
        }

        UCD suggested = parsedUCD.getSuggestion();
        String suggestionString = suggested == null ? null : suggested.toString();
        if (suggestionString != null && !suggestionString.isEmpty() && !suggestionString.equals(parsedUCD.toString())) {
            suggestedUCD = suggestionString;
        } else {
            suggestedUCD = null;
        }
    }

    public ParsedUCD(String ucdText) {
        this(UCDParser.parseUCD(ucdText));
    }

    public boolean isValid() {
        return valid;
    }

    public boolean isRecognised() {
        return recognised;
    }

    public boolean isRecommended() {
        return recommended;
    }

    public List<String> getErrors() {
        return errors;
    }

    public List<String> getAdvices() {
        return advices;
    }

    public String getSuggestedUCD() {
        return suggestedUCD;
    }
}
