/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.oats.ia2.tapschemamanager.webapp;

import it.inaf.oats.ia2.tapschemamanager.api.Credentials;
import it.inaf.oats.ia2.tapschemamanager.api.DBWrapper;
import java.io.IOException;
import java.io.Serializable;
import java.sql.SQLException;
import java.util.List;
import javax.annotation.PostConstruct;
import javax.enterprise.context.Conversation;
import javax.enterprise.context.SessionScoped;
import javax.faces.application.FacesMessage;
import javax.faces.context.FacesContext;
import javax.inject.Inject;
import javax.inject.Named;
import javax.servlet.http.HttpServletRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
@Named("credentialsInsertion")
@SessionScoped
public class CredentialsBean implements Serializable {

    private static final long serialVersionUID = -2688980249773483198L;
    private static final Logger log = LoggerFactory.getLogger(CredentialsBean.class);

    @Inject
    WebAppConfigurationBean ccBean;

    @Inject
    Conversation conversation;

    @Inject
    SchemaSelectionBean schemaSelectionBean;

    private boolean loggedIn;
    private String adminPassword;
    private String loginError;

    private Credentials sourceCredentials;
    private Credentials tapSchemaCredentials;
    private boolean separateCredentials;
    private int currentEditingRow;

    @PostConstruct
    public void init() {
        log.debug("CredentialsBean created");

        if (!conversation.isTransient()) {
            conversation.end();
        }

        sourceCredentials = new Credentials();
        tapSchemaCredentials = new Credentials();
    }

    public void login() {
        if (adminPassword != null && adminPassword.equals(ccBean.getPassword())) {
            loggedIn = true;
        } else {
            FacesContext.getCurrentInstance().addMessage("main:password", new FacesMessage("Invalid credentials"));
        }
    }

    public String getLoginError() {
        return loginError;
    }

    public List getSavedCredentials() {
        return ccBean.getConfig().getCredentialsInfo();
    }

    public void editCredentials(Credentials credentials, int index) {
        this.sourceCredentials = credentials;
        separateCredentials = false;
        currentEditingRow = index;
    }

    public void editSeparateCredentials(SeparateCredentials sc, int index) {
        this.sourceCredentials = sc.getSourceCredentials();
        this.tapSchemaCredentials = sc.getTapSchemaCredentials();
        currentEditingRow = index;
        separateCredentials = true;
    }

    public void addNewCredentials() {
        separateCredentials = false;
        this.sourceCredentials = new Credentials();
        this.tapSchemaCredentials = new Credentials();
        currentEditingRow = ccBean.getConfig().getCredentialsInfo().size();
    }

    public String loginWithSingleCredentials(Credentials credentials) {
        log.debug("Login with single credentials");
        return loginWithDBWrapper(new DBWrapper(credentials));
    }

    public String loginWithSeparatedCredentials(Credentials sourceCredentials, Credentials tapSchemaCredentials) {
        log.debug("Login with separated credentials");
        return loginWithDBWrapper(new DBWrapper(sourceCredentials, tapSchemaCredentials));
    }

    private String loginWithDBWrapper(DBWrapper dbWrapper) {
        loginError = null;

        try {
            dbWrapper.testConnections();
            conversation.setTimeout(30 * 60000L); // 30 minutes
            conversation.begin();
            schemaSelectionBean.setDbWrapper(dbWrapper);
            return "schemaSelection.xhtml?faces-redirect=true";
        } catch (SQLException e) {
            log.error("Exception caught", e);
            loginError = "Connection error: " + e.getMessage();
            return null;
        }
    }

    public void removeCredentials(int index) throws IOException {
        ccBean.getConfig().getCredentialsInfo().remove(index);
        ccBean.updateConfigurationFile();
    }

    public void saveCredentialsEdited() throws IOException {

        List credentialsList = ccBean.getConfig().getCredentialsInfo();

        if (currentEditingRow < credentialsList.size()) {
            credentialsList.remove(currentEditingRow);
        }

        if (separateCredentials) {
            SeparateCredentials sc = new SeparateCredentials(sourceCredentials, tapSchemaCredentials);
            credentialsList.add(currentEditingRow, sc);
        } else {
            credentialsList.add(currentEditingRow, sourceCredentials);
        }

        ccBean.updateConfigurationFile();
    }

    public boolean isSeparateCredentials() {
        return separateCredentials;
    }

    public void setSeparateCredentials(boolean separateCredentials) {
        this.separateCredentials = separateCredentials;
    }

    public Credentials getSourceCredentials() {
        return sourceCredentials;
    }

    public void setSourceCredentials(Credentials sourceCredentials) {
        this.sourceCredentials = sourceCredentials;
    }

    public Credentials getTapSchemaCredentials() {
        return tapSchemaCredentials;
    }

    public void setTapSchemaCredentials(Credentials tapSchemaCredentials) {
        this.tapSchemaCredentials = tapSchemaCredentials;
    }

    public String getAdminPassword() {
        return adminPassword;
    }

    public void setAdminPassword(String adminPassword) {
        this.adminPassword = adminPassword;
    }

    public boolean isLoggedIn() {
        return loggedIn;
    }

    public void setLoggedIn(boolean loggedIn) {
        this.loggedIn = loggedIn;
    }

    public String logout() {
        HttpServletRequest request = (HttpServletRequest) FacesContext.getCurrentInstance().getExternalContext().getRequest();
        if (request.isRequestedSessionIdValid()) {
            request.getSession().invalidate();
        }
        return "index.xhtml?faces-redirect=true";
    }
}
