/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.oats.ia2.tapschemamanager.api;

import it.inaf.oats.ia2.tapschemamanager.api.contract.DatabaseType;
import it.inaf.oats.ia2.tapschemamanager.api.contract.Column;
import it.inaf.oats.ia2.tapschemamanager.api.contract.Key;
import it.inaf.oats.ia2.tapschemamanager.api.contract.KeyColumn;
import it.inaf.oats.ia2.tapschemamanager.api.contract.Schema;
import it.inaf.oats.ia2.tapschemamanager.api.contract.Status;
import it.inaf.oats.ia2.tapschemamanager.api.contract.Table;
import it.inaf.oats.ia2.tapschemamanager.api.contract.TapSchema;
import it.inaf.oats.ia2.tapschemamanager.api.contract.TapSchemaVersion;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import static org.junit.Assert.*;
import org.junit.Ignore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class TestAll {

    private static final Logger log = LoggerFactory.getLogger(TestAll.class);

    private static final int SCHEMAS_COUNT = 2; // minimum value: 2
    private static final int TABLES_COUNT = 3;

    private static List<DBWrapper> dbWrappers;

    public TestAll() {
    }

    @BeforeClass
    public static void setUpClass() throws SQLException {

        dbWrappers = new ArrayList<>();

        // MYSQL
        Credentials mysqlCredentials = new Credentials(DatabaseType.MYSQL);
        mysqlCredentials.setHostname("localhost");
        mysqlCredentials.setUsername("root");
        mysqlCredentials.setPassword("root");

        // POSTGRES
        Credentials postgresCredentials = new Credentials(DatabaseType.POSTGRES);
        postgresCredentials.setHostname("localhost");
        postgresCredentials.setUsername("postgres");
        postgresCredentials.setPassword("pippo");

        DBWrapper dbWrapper = new DBWrapper(mysqlCredentials);
        dbWrapper.testConnections();
        dbWrappers.add(dbWrapper);

        dbWrapper = new DBWrapper(postgresCredentials);
        dbWrapper.testConnections();
        //dbWrappers.add(dbWrapper);

        // Mix!
        //dbWrappers.add(new DBWrapper(mysqlCredentials, postgresCredentials));
        //dbWrappers.add(new DBWrapper(postgresCredentials, mysqlCredentials));
    }

    @AfterClass
    public static void tearDownClass() {
    }

    @Before
    public void setUp() {
    }

    @After
    public void tearDown() {
    }

    private void removeTestingDatabases() throws SQLException {
        for (DBWrapper dbWrapper : dbWrappers) {
            try (Connection sourceConnection = dbWrapper.getSourceConnection();
                    Statement statement = sourceConnection.createStatement()) {
                DatabaseType dbType = dbWrapper.getSourceDatabaseType();

                // Removing keys between schema1 and schema0
                if (dbType == DatabaseType.MYSQL) {
                    try (ResultSet rs = statement.executeQuery("SHOW DATABASES WHERE `Database` = 'sch1'")) {
                        // The constraint can be removed only if sch1 exists
                        if (rs.next()) {
                            statement.executeUpdate("ALTER TABLE sch1.table0 DROP FOREIGN KEY sch0table0id_constraint");
                        }
                    }
                } else if (dbType == DatabaseType.POSTGRES) {
                    try (ResultSet rs = statement.executeQuery("SELECT schema_name FROM information_schema.schemata WHERE schema_name = 'sch1'")) {
                        // The constraint can be removed only if sch1 exists
                        if (rs.next()) {
                            statement.executeUpdate("ALTER TABLE sch1.table0 DROP CONSTRAINT sch0table0id_constraint");
                        }
                    }
                } else {
                    throw new UnsupportedOperationException("Database type " + dbType + " not supported");
                }

                for (int i = 0; i < SCHEMAS_COUNT; i++) {
                    if (dbType == DatabaseType.MYSQL) {
                        statement.executeUpdate("DROP DATABASE IF EXISTS sch" + i);
                    } else if (dbType == DatabaseType.POSTGRES) {
                        statement.executeUpdate("DROP SCHEMA IF EXISTS sch" + i + " CASCADE");
                    } else {
                        throw new UnsupportedOperationException("Database type " + dbType + " not supported");
                    }
                }
            }
            try (Connection tapSchemaConnection = dbWrapper.getTapSchemaConnection();
                    Statement statement = tapSchemaConnection.createStatement()) {
                DatabaseType dbType = dbWrapper.getTapSchemaDatabaseType();
                if (dbType == DatabaseType.MYSQL) {
                    statement.executeUpdate("DROP DATABASE IF EXISTS test_tap_schema");
                } else if (dbType == DatabaseType.POSTGRES) {
                    statement.executeUpdate("DROP SCHEMA IF EXISTS test_tap_schema CASCADE");
                } else {
                    throw new UnsupportedOperationException("Database type " + dbType + " not supported");
                }
            }
        }
    }

    private void setUpTestingDatabases() throws SQLException {
        removeTestingDatabases();

        for (DBWrapper dbWrapper : dbWrappers) {

            DatabaseType dbType = dbWrapper.getSourceDatabaseType();

            try (Connection connection = dbWrapper.getSourceConnection()) {

                try (Statement statement = connection.createStatement()) {

                    if (dbType == DatabaseType.MYSQL) {
                        for (int i = 0; i < SCHEMAS_COUNT; i++) {
                            statement.executeUpdate("CREATE DATABASE sch" + i);
                            for (int j = 0; j < TABLES_COUNT; j++) {
                                statement.executeUpdate("CREATE TABLE sch" + i + ".table" + j + " (\n"
                                        + "id INT PRIMARY KEY AUTO_INCREMENT,\n"
                                        + "value1 VARCHAR(255),\n"
                                        + "value2 FLOAT\n"
                                        + ");");
                            }
                        }
                    } else if (dbType == DatabaseType.POSTGRES) {
                        for (int i = 0; i < SCHEMAS_COUNT; i++) {
                            statement.executeUpdate("CREATE SCHEMA sch" + i);
                            for (int j = 0; j < TABLES_COUNT; j++) {
                                statement.executeUpdate("CREATE TABLE sch" + i + ".table" + j + " (\n"
                                        + "id BIGSERIAL PRIMARY KEY,\n"
                                        + "value1 VARCHAR(255),\n"
                                        + "value2 FLOAT\n"
                                        + ");");
                            }
                        }
                    } else {
                        throw new UnsupportedOperationException("Database type " + dbType + " not supported");
                    }
                }

                log.info("dbs created");

                try (Statement statement = connection.createStatement()) {
                    for (int j = 0; j < TABLES_COUNT - 1; j++) {
                        String update1 = "ALTER TABLE sch0.table" + (j + 1) + " ADD COLUMN table" + j + "_id INT";
                        statement.executeUpdate(update1);
                        String update2;
                        if (dbType == DatabaseType.MYSQL || dbType == DatabaseType.POSTGRES) {
                            update2 = "ALTER TABLE sch0.table" + (j + 1) + " ADD CONSTRAINT id_constraint_" + (j + 1) + " FOREIGN KEY (table" + j + "_id) REFERENCES sch0.table" + j + "(id)";
                        } else {
                            throw new UnsupportedOperationException("Database type " + dbType + " not supported");
                        }

                        statement.executeUpdate(update2);
                    }

                    // Foreign key with multiple columns
                    statement.executeUpdate("CREATE TABLE sch0.table_x (idx1 INT, idx2 INT)");
                    statement.executeUpdate("ALTER TABLE sch0.table_x ADD CONSTRAINT pkx PRIMARY KEY (idx1, idx2)");
                    statement.executeUpdate("CREATE TABLE sch0.table_y (idy1 INT, idy2 INT)");
                    statement.executeUpdate("ALTER TABLE sch0.table_y ADD CONSTRAINT pky PRIMARY KEY (idy1, idy2)");
                    statement.executeUpdate("ALTER TABLE sch0.table_y ADD CONSTRAINT fky FOREIGN KEY(idy1, idy2) REFERENCES sch0.table_x(idx1, idx2)");

                    // Foreign keys between different schemas
                    statement.executeUpdate("ALTER TABLE sch0.table0 ADD COLUMN sch1table0id INT");
                    statement.executeUpdate("ALTER TABLE sch0.table0 ADD CONSTRAINT sch1table0id_constraint FOREIGN KEY(sch1table0id) REFERENCES sch1.table0(id)");
                    statement.executeUpdate("ALTER TABLE sch1.table0 ADD COLUMN sch0table0id INT");
                    statement.executeUpdate("ALTER TABLE sch1.table0 ADD CONSTRAINT sch0table0id_constraint FOREIGN KEY(sch0table0id) REFERENCES sch0.table0(id)");
                }
            }
        }
    }

    private boolean allKeysHaveDifferentId(TapSchema tapSchema) {
        boolean differentKeySameId = false;
        Map<String, Key> keys = new HashMap<>();

        for (Key key : ((TapSchemaImpl) tapSchema).getAllKeys()) {
            if (key.getId() != null) {
                if (keys.get(key.getId()) != null && !key.equals(keys.get(key.getId()))) {
                    differentKeySameId = true;
                }
                keys.put(key.getId(), key);
            }
        }

        if (differentKeySameId) {
            log.debug("Found different keys with the same key_id!");
            for (Key key : ((TapSchemaImpl) tapSchema).getAllKeys()) {
                log.debug(key.toString());
            }
        }

        return !differentKeySameId;
    }

    private void checkKey(Key key, String fromTableCompleteName, String[] fromColumns, String targetTableCompleteName, String[] targetColumns, boolean isVisible) {
        assertEquals(fromTableCompleteName, key.getFromTableCompleteName());
        assertEquals(targetTableCompleteName, key.getTargetTableCompleteName());
        for (int i = 0; i < key.getKeyColumns().size(); i++) {
            KeyColumn keyColumn = key.getKeyColumns().get(i);
            assertEquals(fromColumns[i], keyColumn.getFromColumn());
            assertEquals(targetColumns[i], keyColumn.getTargetColumn());
        }
        assertEquals(isVisible, key.isVisible());
        for (KeyColumn keyColumn : key.getKeyColumns()) {
            assertEquals(key.getId(), keyColumn.getKeyId());
        }
    }

    private void checkKey(Key key, String fromTableCompleteName, String fromColumn, String targetTableCompleteName, String targetColumn, boolean isVisible) {
        checkKey(key, fromTableCompleteName, new String[]{fromColumn}, targetTableCompleteName, new String[]{targetColumn}, isVisible);
    }

    @Test
    public void createNewAndUpdate() throws SQLException {
        log.info("TEST createNewAndUpdate STARTED");

        try {
            removeTestingDatabases();

            setUpTestingDatabases();

            for (DBWrapper dbWrapper : dbWrappers) {

                // Initializing a not existing TAP_SCHEMA
                TapSchema tapSchema = TapSchemaFactory.getTapSchema(TapSchemaVersion.TAP_SCHEMA_1_IA2, dbWrapper, "test_tap_schema", false);

                /////////////////////////////////////
                //         ADDING A SCHEMA         //
                /////////////////////////////////////
                //
                Schema sch0 = tapSchema.addChild("sch0");
                assertEquals(Status.ADDED_NOT_PERSISTED, sch0.getStatus());

                Set<Key> allKeys = ((TapSchemaImpl) tapSchema).getAllKeys();
                log.debug("ALL keys:");
                for (Key key : allKeys) {
                    log.debug(key.toString());
                }

                // In the testing schemas each numbered table references the id 
                // of the previous table, except for the first table, so there
                // should be "TABLES_COUNT - 1" keys for the numbered tables.
                // Moreover, there are also the following keys:
                // - sch0.table0.sch1table0id -> sch1.table0.id
                // - sch1.table0.sch0table0id -> sch0.table0.id
                // - sch0.table_y.(idy1, idy2) -> sch0.table_x.(idyx, idx2)
                // so we check for TABLES_COUNT + 2.
                assertEquals(TABLES_COUNT + 2, allKeys.size());

                // Checking that keys information has been filled correctly.
                for (Key schemaKey : allKeys) {
                    assertFalse(schemaKey.isVisible());
                    assertNull(schemaKey.getId());
                    assertEquals(schemaKey.getFromTableCompleteName(), schemaKey.getFromSchemaName() + "." + schemaKey.getFromTableSimpleName());
                    assertEquals(schemaKey.getTargetTableCompleteName(), schemaKey.getTargetSchemaName() + "." + schemaKey.getTargetTableSimpleName());
                    assertTrue(schemaKey.getKeyColumns().size() >= 1);
                    for (KeyColumn keyColumn : schemaKey.getKeyColumns()) {
                        assertNotNull(keyColumn.getFromColumn());
                        assertNotNull(keyColumn.getTargetColumn());
                    }
                }

                /////////////////////////////////////
                //         ADDING A TABLE          //
                /////////////////////////////////////
                //
                Table sch0table0 = sch0.addChild("table0");
                assertEquals(Status.ADDED_NOT_PERSISTED, sch0table0.getStatus());
                assertEquals("sch0.table0", sch0table0.getCompleteName());
                assertEquals(1, sch0.getChildren().size());

                assertEquals(4, sch0table0.getAddableChildrenNames().size());

                Column sch0table0id = sch0table0.addChild("id");
                assertEquals(Status.ADDED_NOT_PERSISTED, sch0table0id.getStatus());

                // Primary key check
                assertTrue(sch0table0id.isPrimaryKey());

                /////////////////////////////////////
                //         KEYS MANAGEMENT         //
                /////////////////////////////////////
                //
                // CASE 1: Foreign key between two tables in the same schema.
                // sch0.table1.t0id -> sch0.table0.id
                //
                // Adding sch0.table1
                Table sch0table1 = sch0.addChild("table1");
                assertEquals(2, sch0.getChildren().size());

                assertTrue(sch0table0.getVisibleFromKeys().isEmpty());
                assertTrue(sch0table0.getVisibleTargetKeys().isEmpty());
                assertTrue(sch0table1.getVisibleFromKeys().isEmpty());
                assertTrue(sch0table1.getVisibleTargetKeys().isEmpty());

                sch0table1.addChild("table0_id");
                // Now sch0table0.getVisibleTargetKeys() and sch0table1.getVisibleFromKeys()
                // have to return the same Key object with id = 1

                assertTrue(sch0table0.getVisibleFromKeys().isEmpty());
                assertEquals(1, sch0table0.getVisibleTargetKeys().size());
                assertEquals(1, sch0table1.getVisibleFromKeys().size());
                assertTrue(sch0table1.getVisibleTargetKeys().isEmpty());

                // Check if key and its columns have been properly initialized
                Key sch0table0TargetKey = sch0table0.getVisibleTargetKeys().get(0);
                Key sch0table1TargetKey = sch0table1.getVisibleFromKeys().get(0);
                checkKey(sch0table0TargetKey, "sch0.table1", "table0_id", "sch0.table0", "id", true);
                assertEquals("1", sch0table0TargetKey.getId());
                assertEquals(sch0table0TargetKey, sch0table1TargetKey);

                // Removing sch0.table1
                assertEquals(1, sch0table0.getVisibleTargetKeys().size()); // sch0.table1.table0_id -> sch0.table0.id
                sch0.removeChild("table1");
                assertEquals(1, sch0.getAddedChildren().size());
                assertTrue(sch0table0.getVisibleFromKeys().isEmpty());
                assertTrue(sch0table0.getVisibleTargetKeys().isEmpty());
                assertFalse(sch0table0TargetKey.isVisible());

                //
                // CASE 2: Foreign key between two tables in different schemas.
                // sch1.table0.sch0table0id -> sch0.table0.id
                //
                // Adding sch1
                Schema sch1 = tapSchema.addChild("sch1");
                allKeys = ((TapSchemaImpl) tapSchema).getAllKeys();
                assertEquals(5, allKeys.size());

                // Adding sch1.table0
                Table sch1table0 = sch1.addChild("table0");

                sch1table0.addChild("id");
                sch1table0.addChild("sch0table0id"); // sch1.table0.sch0table0id -> sch0.table0.it obtains keyId = "1"
                sch0table0.addChild("sch1table0id");// sch0.table0.sch1table0id -> sch1.table0.it obtains keyId = "2"

                assertEquals(0, sch0table1.getVisibleFromKeys().size());
                assertEquals(1, sch0table0.getVisibleFromKeys().size());
                assertEquals(1, sch0table0.getVisibleTargetKeys().size());
                assertEquals(1, sch1table0.getVisibleFromKeys().size());
                assertEquals(1, sch1table0.getVisibleTargetKeys().size());

                for (Key key : allKeys) {
                    if (key.getId() == null) {
                        assertFalse(key.isVisible());
                    } else {
                        switch (key.getId()) {
                            case "1":
                                checkKey(key, "sch0.table1", "table0_id", "sch0.table0", "id", false);
                                break;
                            case "2":
                                checkKey(key, "sch1.table0", "sch0table0id", "sch0.table0", "id", true);
                                break;
                            case "3":
                                checkKey(key, "sch0.table0", "sch1table0id", "sch1.table0", "id", true);
                                break;
                        }
                    }
                }

                assertTrue(allKeysHaveDifferentId(tapSchema));

                // Removing sch1
                tapSchema.removeChild("sch1");
                assertEquals(Status.REMOVED_NOT_PERSISTED, sch1.getStatus());
                assertTrue(sch0table0.getVisibleTargetKeys().isEmpty());

                // Case 2B: Re-adding sch1
                // sch1.table0 has not been removed from its schema, so the keys
                // should be re-added.
                tapSchema.addChild("sch1");
                assertEquals(Status.ADDED_NOT_PERSISTED, sch1.getStatus());
                assertEquals(1, sch0table0.getVisibleTargetKeys().size());
                assertEquals(1, sch1table0.getVisibleFromKeys().size());

                for (Key key : allKeys) {
                    if (key.getId() == null) {
                        assertFalse(key.isVisible());
                    } else {
                        switch (key.getId()) {
                            case "1":
                                checkKey(key, "sch0.table1", "table0_id", "sch0.table0", "id", false);
                                break;
                            case "2":
                                checkKey(key, "sch1.table0", "sch0table0id", "sch0.table0", "id", true);
                                break;
                            case "3":
                                checkKey(key, "sch0.table0", "sch1table0id", "sch1.table0", "id", true);
                                break;
                        }
                    }
                }

                //
                // CASE 3: foreign key with multiple columns
                //
                Table table_x = sch0.addChild("table_x");
                Table table_y = sch0.addChild("table_y");

                assertTrue(table_x.getVisibleFromKeys().isEmpty());
                assertTrue(table_x.getVisibleTargetKeys().isEmpty());
                assertTrue(table_y.getVisibleFromKeys().isEmpty());
                assertTrue(table_y.getVisibleTargetKeys().isEmpty());

                table_x.addChild("idx1");
                table_x.addChild("idx2");
                table_y.addChild("idy1");

                assertTrue(table_x.getVisibleFromKeys().isEmpty());
                assertTrue(table_x.getVisibleTargetKeys().isEmpty());
                assertTrue(table_y.getVisibleFromKeys().isEmpty());
                assertTrue(table_y.getVisibleTargetKeys().isEmpty());

                table_y.addChild("idy2");

                assertEquals(1, table_y.getVisibleFromKeys().size());
                assertEquals(1, table_x.getVisibleTargetKeys().size());
                assertTrue(table_x.getVisibleFromKeys().isEmpty());
                assertTrue(table_y.getVisibleTargetKeys().isEmpty());

                for (Key key : allKeys) {
                    if (key.getId() == null) {
                        assertFalse(key.isVisible());
                    } else {
                        switch (key.getId()) {
                            case "1":
                                checkKey(key, "sch0.table1", "table0_id", "sch0.table0", "id", false);
                                break;
                            case "2":
                                checkKey(key, "sch1.table0", "sch0table0id", "sch0.table0", "id", true);
                                break;
                            case "3":
                                checkKey(key, "sch0.table0", "sch1table0id", "sch1.table0", "id", true);
                                break;
                            case "4":
                                checkKey(key, "sch0.table_y", new String[]{"idy1", "idy2"}, "sch0.table_x", new String[]{"idx1", "idx2"}, true);
                                break;
                        }
                    }
                }

                assertTrue(allKeysHaveDifferentId(tapSchema));

                /////////////////////////////////////
                //              SAVE               //
                /////////////////////////////////////
                sch0table1 = sch0.addChild("table1");
                sch0table1.addChild("id");
                Table sch0table2 = sch0.addChild("table2");
                sch0table2.addChild("table1_id");

                for (Key key : allKeys) {
                    if (key.getId() == null) {
                        assertFalse(key.isVisible());
                    } else {
                        switch (key.getId()) {
                            case "1":
                                checkKey(key, "sch0.table1", "table0_id", "sch0.table0", "id", true);
                                break;
                            case "2":
                                checkKey(key, "sch1.table0", "sch0table0id", "sch0.table0", "id", true);
                                break;
                            case "3":
                                checkKey(key, "sch0.table0", "sch1table0id", "sch1.table0", "id", true);
                                break;
                            case "4":
                                checkKey(key, "sch0.table_y", new String[]{"idy1", "idy2"}, "sch0.table_x", new String[]{"idx1", "idx2"}, true);
                                break;
                            case "5":
                                checkKey(key, "sch0.table2", "table1_id", "sch0.table1", "id", true);
                                break;
                        }
                    }
                }
                assertTrue(allKeysHaveDifferentId(tapSchema));

                UpdateOperations operations = new UpdateOperations(tapSchema);
                assertEquals(2, operations.getSchemasToAdd().size());
                assertEquals(6, operations.getTablesToAdd().size());
                assertEquals(11, operations.getColumnsToAdd().size());
                assertEquals(5, operations.getKeysToAdd().size());

                tapSchema.save();
                assertFalse(new UpdateOperations(tapSchema).getHasOperations());
                assertTrue(allKeysHaveDifferentId(tapSchema));

                // reloading
                log.debug("----- Reloading saved TAP_SCHEMA -----");
                tapSchema = TapSchemaFactory.getTapSchema(TapSchemaVersion.TAP_SCHEMA_1_IA2, dbWrapper, "test_tap_schema", true);
                allKeys = ((TapSchemaImpl) tapSchema).getAllKeys();
                assertTrue(allKeysHaveDifferentId(tapSchema));
                log.debug(tapSchema.toString());

                assertNotNull(sch0 = tapSchema.getChild("sch0", Status.ADDED_PERSISTED));
                assertNotNull(sch1 = tapSchema.getChild("sch1", Status.ADDED_PERSISTED));
                assertNotNull(sch0table0 = sch0.getChild("table0", Status.ADDED_PERSISTED));
                assertNotNull(sch0table1 = sch0.getChild("table1", Status.ADDED_PERSISTED));
                assertNotNull(sch0table2 = sch0.getChild("table2", Status.ADDED_PERSISTED));
                assertNotNull(table_x = sch0.getChild("table_x", Status.ADDED_PERSISTED));
                assertNotNull(table_y = sch0.getChild("table_y", Status.ADDED_PERSISTED));
                assertNotNull(sch1table0 = sch1.getChild("table0", Status.ADDED_PERSISTED));
                assertNotNull(sch0table0.getChild("id", Status.ADDED_PERSISTED));
                assertNotNull(sch0table0.getChild("sch1table0id", Status.ADDED_PERSISTED));
                assertNotNull(sch0table1.getChild("id", Status.ADDED_PERSISTED));
                assertNotNull(sch0table1.getChild("table0_id", Status.ADDED_PERSISTED));
                assertNotNull(sch0table2.getChild("table1_id", Status.ADDED_PERSISTED));
                assertNotNull(table_x.getChild("idx1", Status.ADDED_PERSISTED));
                assertNotNull(table_x.getChild("idx2", Status.ADDED_PERSISTED));
                assertNotNull(table_y.getChild("idy1", Status.ADDED_PERSISTED));
                assertNotNull(table_y.getChild("idy2", Status.ADDED_PERSISTED));
                assertNotNull(sch1table0.getChild("id", Status.ADDED_PERSISTED));
                assertNotNull(sch1table0.getChild("sch0table0id", Status.ADDED_PERSISTED));

                for (Key key : allKeys) {
                    if (key.getId() == null) {
                        assertFalse(key.isVisible());
                    } else {
                        switch (key.getId()) {
                            case "1":
                                checkKey(key, "sch0.table1", "table0_id", "sch0.table0", "id", true);
                                break;
                            case "2":
                                checkKey(key, "sch1.table0", "sch0table0id", "sch0.table0", "id", true);
                                break;
                            case "3":
                                checkKey(key, "sch0.table0", "sch1table0id", "sch1.table0", "id", true);
                                break;
                            case "4":
                                checkKey(key, "sch0.table_y", new String[]{"idy1", "idy2"}, "sch0.table_x", new String[]{"idx1", "idx2"}, true);
                                break;
                            case "5":
                                checkKey(key, "sch0.table2", "table1_id", "sch0.table1", "id", true);
                                break;
                        }
                    }
                }

                List<Key> sch0table1FromKeys = sch0table1.getVisibleFromKeys();
                assertEquals(1, sch0table1FromKeys.size());
                assertEquals(1, sch0table1.getVisibleTargetKeys().size());

                sch0.removeChild("table1");
                assertEquals(0, sch0table1.getVisibleFromKeys().size());

                for (Key key : allKeys) {
                    if (key.getId() == null) {
                        assertFalse(key.isVisible());
                    } else {
                        switch (key.getId()) {
                            case "1":
                                checkKey(key, "sch0.table1", "table0_id", "sch0.table0", "id", false);
                                break;
                        }
                    }
                }

                operations = new UpdateOperations(tapSchema);
                assertFalse(operations.getHasEntitiesToUpdate());
                assertFalse(operations.getHasEntitiesToAdd());
                assertEquals(0, operations.getSchemasToRemove().size());
                assertEquals(1, operations.getTablesToRemove().size());
                assertEquals(2, operations.getColumnsToRemove().size());
                assertEquals(2, operations.getKeysToRemove().size());

                tapSchema.save();
                log.debug(tapSchema.toString());
                assertFalse(new UpdateOperations(tapSchema).getHasOperations());

                // reloading
                log.debug("----- Reloading saved TAP_SCHEMA -----");
                tapSchema = TapSchemaFactory.getTapSchema(TapSchemaVersion.TAP_SCHEMA_1_IA2, dbWrapper, "test_tap_schema", true);
                allKeys = ((TapSchemaImpl) tapSchema).getAllKeys();
                log.debug(tapSchema.toString());

                assertNotNull(sch0 = tapSchema.getChild("sch0", Status.ADDED_PERSISTED));
                assertNotNull(sch1 = tapSchema.getChild("sch1", Status.ADDED_PERSISTED));
                assertNotNull(sch0table0 = sch0.getChild("table0", Status.ADDED_PERSISTED));
                assertNotNull(sch0table2 = sch0.getChild("table2", Status.ADDED_PERSISTED));
                assertNotNull(table_x = sch0.getChild("table_x", Status.ADDED_PERSISTED));
                assertNotNull(table_y = sch0.getChild("table_y", Status.ADDED_PERSISTED));
                assertNotNull(sch1table0 = sch1.getChild("table0", Status.ADDED_PERSISTED));
                assertNotNull(sch0table0.getChild("id", Status.ADDED_PERSISTED));
                assertNotNull(sch0table0.getChild("sch1table0id", Status.ADDED_PERSISTED));
                assertNotNull(sch0table2.getChild("table1_id", Status.ADDED_PERSISTED));
                assertNotNull(table_x.getChild("idx1", Status.ADDED_PERSISTED));
                assertNotNull(table_x.getChild("idx2", Status.ADDED_PERSISTED));
                assertNotNull(table_y.getChild("idy1", Status.ADDED_PERSISTED));
                assertNotNull(table_y.getChild("idy2", Status.ADDED_PERSISTED));
                assertNotNull(sch1table0.getChild("id", Status.ADDED_PERSISTED));
                assertNotNull(sch1table0.getChild("sch0table0id", Status.ADDED_PERSISTED));

                for (Key key : allKeys) {
                    if (key.getId() == null) {
                        assertFalse(key.isVisible());
                    } else {
                        assertNotNull(key.getOriginalValue(Key.ID_KEY, String.class)); // for reloaded keys
                        switch (key.getId()) {
                            case "2":
                                checkKey(key, "sch1.table0", "sch0table0id", "sch0.table0", "id", true);
                                break;
                            case "3":
                                checkKey(key, "sch0.table0", "sch1table0id", "sch1.table0", "id", true);
                                break;
                            case "4":
                                checkKey(key, "sch0.table_y", new String[]{"idy1", "idy2"}, "sch0.table_x", new String[]{"idx1", "idx2"}, true);
                                break;
                            case "5":
                                checkKey(key, "sch0.table2", "table1_id", "sch0.table1", "id", true);
                                break;
                        }
                    }
                }

                // Test adding ficitious key
                sch0table0.addChild("value1");
                sch0table2.addChild("value1");
                ((TapSchemaImpl) tapSchema).addFictitiousKey(sch0table2, new String[]{"value1"}, sch0table0, new String[]{"value1"});
                operations = new UpdateOperations(tapSchema);
                assertEquals(1, operations.getKeysToAdd().size());
                tapSchema.save();

                tapSchema = TapSchemaFactory.getTapSchema(TapSchemaVersion.TAP_SCHEMA_1_IA2, dbWrapper, "test_tap_schema", true);

                sch0table2 = tapSchema.getChild("sch0").getChild("table2");
                assertEquals(1, sch0table2.getVisibleFromKeys().size());
                checkKey(sch0table2.getVisibleFromKeys().get(0), "sch0.table2", "value1", "sch0.table0", "value1", true);
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            //removeTestingDatabases(credentials);
        }
    }

    @Test
    public void testTapSchemaSerialization() throws Exception {
        for (DBWrapper dbWrapper : dbWrappers) {
            if (dbWrapper.getTapSchemaDatabaseType() == DatabaseType.MYSQL) { // currently "tng_TAP_SCHEMA" exists in a MySQL instance
                TapSchema tapSchema = TapSchemaFactory.getTapSchema(TapSchemaVersion.TAP_SCHEMA_1_IA2, dbWrapper, "test_tap_schema", true);

                File temp = File.createTempFile("test_tap_schema", ".ser");

                try (FileOutputStream fileOut = new FileOutputStream(temp);
                        ObjectOutputStream out = new ObjectOutputStream(fileOut)) {
                    out.writeObject(tapSchema);
                }

                try (FileInputStream fileIn = new FileInputStream(temp);
                        ObjectInputStream in = new ObjectInputStream(fileIn)) {
                    tapSchema = (TapSchema) in.readObject();
                }

                log.debug(tapSchema.toString());
            }
        }
    }
}
