/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.oats.ia2.tapschemamanager.api.contract;

import java.util.List;

/**
 * Represents an object that is mapped on a table of the TAP_SCHEMA.
 *
 * A TapSchemaEntity has some properties that correspond to columns of the table
 * represented by the TapSchemaEntity.<br>
 * Property value can be changed but the original value has to be maintained
 * until the {@link #save()} method is called.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public interface TapSchemaEntity {

    /**
     * Initializes the value of a property (store the original value).
     */
    <T> void initProperty(String key, T value);

    /**
     * Returns true if one or more property values is changed (the current value
     * is different from the original value).
     */
    boolean isChanged();

    /**
     * Returns true the value of property which key is passed as parameter is
     * changed (the current value is different from the original value).
     */
    boolean isChanged(String key);

    /**
     * Retrieve the current value of the property (the last value set).
     */
    <T> T getValue(String key, Class<T> type);

    /**
     * Retrieve the original value of the property.
     *
     * @param key the name of the property (the name of the table column).
     * @param type the class of the property value.
     * @return
     */
    <T> T getOriginalValue(String key, Class<T> type);

    /**
     * Retrieve a list of all properties names (the names of the table columns).
     */
    List<String> getPropertiesKeys();

    /**
     * Marks the TapSchemaEntity as saved (all original values are set equals to
     * current values).
     */
    void save();
}
