/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.oats.ia2.tapschemamanager.api;

import it.inaf.oats.ia2.tapschemamanager.api.contract.DatabaseType;
import it.inaf.oats.ia2.tapschemamanager.api.contract.TapSchema;
import it.inaf.oats.ia2.tapschemamanager.api.contract.TapSchemaEntity;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Prepares an {@code UPDATE } SQL query for a given {@link TapSchemaEntity} and
 * a given {@link TapSchema}.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class UpdateQueryBuilder {

    private static final Logger log = LoggerFactory.getLogger(UpdateQueryBuilder.class);

    private final String query;
    private final List<EntityPropertyInfo> addedProperties;
    private final TapSchemaEntity tapSchemaEntity;

    protected UpdateQueryBuilder(DatabaseType dbType, TapSchema tapSchema, TapSchemaEntity tapSchemaEntity, String tapSchemaTableName, String whereCondition) {

        StringBuilder querySb = new StringBuilder("UPDATE ");
        querySb.append(TSMUtil.escapeName(tapSchema.getName(), dbType));
        querySb.append(".");
        querySb.append(TSMUtil.escapeName(tapSchemaTableName, dbType));
        querySb.append("\nSET");

        addedProperties = new ArrayList<>();
        this.tapSchemaEntity = tapSchemaEntity;

        boolean first = true;
        for (EntityPropertyInfo propertyInfo : EntityPropertyInfo.getEntityPropertiesInfo(tapSchemaTableName)) {
            if (propertyInfo.isUpdatable()
                    && (propertyInfo.acceptVersion(tapSchema.getVersion()))
                    && tapSchemaEntity.isChanged(propertyInfo.getPropertyKey())) {

                if (!first) {
                    querySb.append(",");
                }
                querySb.append(" ");
                querySb.append(propertyInfo.getPropertyKey());
                querySb.append(" = ?");
                addedProperties.add(propertyInfo);
                first = false;
            }
        }

        querySb.append("\nWHERE ");
        querySb.append(whereCondition);

        query = querySb.toString();
    }

    public String getQuery() {
        return query;
    }

    protected int addStatementValues(PreparedStatement statement) throws SQLException {
        int i = 1;
        for (EntityPropertyInfo property : addedProperties) {
            Object value = tapSchemaEntity.getValue(property.getPropertyKey(), property.getPropertyType());
            statement.setObject(i, value, property.getSqlType());
            log.debug("\t[{}] {}", property.getPropertyKey(), value);
            i++;
        }
        return i;
    }
}
