/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.oats.ia2.tapschemamanager.api;

import it.inaf.oats.ia2.tapschemamanager.api.contract.Column;
import it.inaf.oats.ia2.tapschemamanager.api.contract.Key;
import it.inaf.oats.ia2.tapschemamanager.api.contract.Schema;
import it.inaf.oats.ia2.tapschemamanager.api.contract.Status;
import it.inaf.oats.ia2.tapschemamanager.api.contract.Table;
import it.inaf.oats.ia2.tapschemamanager.api.contract.TapSchema;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The main implementation of {@link Table}.
 * 
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class TableImpl extends ChildEntityImpl<Schema> implements Table {

    private static final long serialVersionUID = 8265331530960896871L;
    private static final Logger log = LoggerFactory.getLogger(TableImpl.class);

    private final Map<String, Column> columns;
    private final List<Key> fromKeys;
    private final List<Key> targetKeys;

    private String simpleName;
    private Schema parentSchema;

    private TableImpl() {
        // for serialization
        super();
        columns = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);
        fromKeys = new ArrayList<>();
        targetKeys = new ArrayList<>();
    }

    protected TableImpl(DBWrapper dbWrapper, TapSchema tapSchema, Schema schema, String tableSimpleName, String tableType) throws SQLException {
        super(dbWrapper, tapSchema);
        parentSchema = schema;
        String schemaName = schema.getName();
        this.simpleName = tableSimpleName;
        columns = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);

        addProperty(SCHEMA_NAME_KEY, new FixedEntityProperty<>(schemaName));
        addProperty(TABLE_NAME_KEY, new FixedEntityProperty<>(schemaName + "." + tableSimpleName));
        addProperty(TABLE_TYPE_KEY, new FixedEntityProperty<>(tableType));

        addProperty(UTYPE_KEY, new EditableProperty<String>());
        addProperty(DESCRIPTION_KEY, new EditableProperty<String>());
        addProperty(TABLE_INDEX, new EditableProperty<Integer>());
        addProperty(TABLE_ID, new EditableProperty<Long>());

        fromKeys = new ArrayList<>();
        targetKeys = new ArrayList<>();

        for (Column column : DaoColumn.loadAllTableColumns(dbWrapper, tapSchema, schemaName, (this))) {
            columns.put(column.getName(), column);
        }

        setStatus(Status.LOADED);
    }

    /**
     * Only the table name.
     */
    @Override
    public String getName() {
        return simpleName;
    }

    /**
     * {@code schema_name.table_name}.
     */
    @Override
    public String getCompleteName() {
        return getValue(TABLE_NAME_KEY, String.class);
    }

    /**
     * {@inheritDoc }
     */
    @Override
    public Column addChild(String columnName) {
        String tableCompleteName = getCompleteName();

        log.debug("Adding column {} into table {}", columnName, tableCompleteName);

        if (!columns.containsKey(columnName)) {
            throw new IllegalArgumentException("Column " + columnName + " doesn't exists in table " + tableCompleteName);
        }

        Column column = columns.get(columnName);

        switch (column.getStatus()) {
            case LOADED:
                column.setStatus(Status.ADDED_NOT_PERSISTED);
                break;
            case REMOVED_NOT_PERSISTED: // undo removal
                column.setStatus(Status.ADDED_NOT_PERSISTED);
                break;
            case TO_REMOVE: // undo removal
                column.setStatus(Status.ADDED_PERSISTED);
                break;
            default:
                throw new IllegalArgumentException("Cannot add column " + columnName + " in table " + tableCompleteName + ". Invalid column status: " + column.getStatus());
        }

        ((TapSchemaImpl) tapSchema).checkKeys();

        return column;
    }

    /**
     * {@inheritDoc }
     */
    @Override
    public void removeChild(String columnName) {
        String tableCompleteName = getCompleteName();

        log.debug("Removing column {} from table {}", columnName, tableCompleteName);

        if (!columns.containsKey(columnName)) {
            throw new IllegalArgumentException("Column " + columnName + " doesn't exists in table " + tableCompleteName);
        }

        Column column = columns.get(columnName);

        switch (column.getStatus()) {
            case ADDED_NOT_PERSISTED:
                column.setStatus(Status.REMOVED_NOT_PERSISTED);
                break;
            case ADDED_PERSISTED:
                column.setStatus(Status.TO_REMOVE);
                break;
            default:
                throw new IllegalArgumentException("Cannot remove column " + columnName + " in table " + tableCompleteName + ". Invalid column status: " + column.getStatus());
        }

        ((TapSchemaImpl) tapSchema).checkKeys();
    }

    /**
     * {@inheritDoc }
     */
    @Override
    public Column getChild(String childName, Status... statuses) {
        return TSMUtil.getChild(columns, childName, statuses);
    }

    /**
     * {@inheritDoc }
     */
    @Override
    public List<Column> getChildren(Status... statuses) {
        return TSMUtil.getChildrenByStatus(columns.values(), statuses);
    }

    /**
     * {@inheritDoc }
     */
    @Override
    public List<String> getAddableChildrenNames() {
        return TSMUtil.getAddableChildrenNames(columns);
    }

    @Override
    public List<Column> getAddedChildren() {
        return getChildren(Status.ADDED_PERSISTED, Status.ADDED_NOT_PERSISTED);
    }

    @Override
    public List<Column> getAddedOrRemovedChildren() {
        return getChildren(Status.ADDED_PERSISTED, Status.ADDED_NOT_PERSISTED, Status.TO_REMOVE, Status.REMOVED_NOT_PERSISTED);
    }

    @Override
    public String getTableType() {
        return getValue(TABLE_TYPE_KEY, String.class);
    }

    @Override
    public Integer getTableIndex() {
        return getValue(TABLE_INDEX, Integer.class);
    }

    @Override
    public void setTableIndex(Integer tableIndex) {
        setValue(TABLE_INDEX, tableIndex);
    }

    @Override
    public Long getTableID() {
        return getValue(TABLE_ID, Long.class);
    }

    @Override
    public void setTableID(Long tableID) {
        setValue(TABLE_ID, tableID);
    }

    @Override
    public List<Key> getVisibleFromKeys() {
        List<Key> ret = new ArrayList<>();
        for (Key key : fromKeys) {
            if (key.isVisible()) {
                ret.add(key);
            }
        }
        return Collections.unmodifiableList(ret);
    }

    @Override
    public List<Key> getAllFromKeys() {
        return Collections.unmodifiableList(fromKeys);
    }

    @Override
    public List<Key> getVisibleTargetKeys() {
        List<Key> ret = new ArrayList<>();
        for (Key key : targetKeys) {
            if (key.isVisible()) {
                ret.add(key);
            }
        }
        return Collections.unmodifiableList(ret);
    }

    @Override
    public List<Key> getAllTargetKeys() {
        return Collections.unmodifiableList(targetKeys);
    }

    @Override
    public void addFromKey(Key key) {
        if (!fromKeys.contains(key)) {
            fromKeys.add(key);
        }
    }

    @Override
    public void addTargetKey(Key key) {
        if (!targetKeys.contains(key)) {
            targetKeys.add(key);
        }
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 29 * hash + Objects.hashCode(this.getCompleteName());
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final TableImpl other = (TableImpl) obj;
        if (!Objects.equals(this.getCompleteName(), other.getCompleteName())) {
            return false;
        }
        return true;
    }

    @Override
    public String getSchemaName() {
        return getValue(SCHEMA_NAME_KEY, String.class);
    }

    @Override
    public String getUtype() {
        return getValue(UTYPE_KEY, String.class);
    }

    @Override
    public void setUtype(String utype) {
        setValue(UTYPE_KEY, utype);
    }

    @Override
    public String getDescription() {
        return getValue(DESCRIPTION_KEY, String.class);
    }

    @Override
    public void setDescription(String description) {
        setValue(DESCRIPTION_KEY, description);
    }

    protected void afterUpdate() {
        for (Column column : columns.values()) {
            if (column.getStatus() == Status.ADDED_NOT_PERSISTED) {
                column.setStatus(Status.ADDED_PERSISTED);
            } else if (column.getStatus() != Status.ADDED_PERSISTED) {
                column.setStatus(Status.LOADED);
            }
        }
    }

    @Override
    public Schema getParent() {
        return parentSchema;
    }
}
