/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.oats.ia2.tapschemamanager.api;

import it.inaf.oats.ia2.tapschemamanager.api.contract.Column;
import it.inaf.oats.ia2.tapschemamanager.api.contract.Key;
import it.inaf.oats.ia2.tapschemamanager.api.contract.KeyColumn;
import it.inaf.oats.ia2.tapschemamanager.api.contract.Schema;
import it.inaf.oats.ia2.tapschemamanager.api.contract.Table;
import it.inaf.oats.ia2.tapschemamanager.api.contract.TapSchema;
import it.inaf.oats.ia2.tapschemamanager.api.contract.TapSchemaVersion;
import java.sql.Types;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Describes the mapping between the
 * {@link it.inaf.oats.ia2.tapschemamanager.api.contract.TapSchemaEntity} instances
 * and their related table columns.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class EntityPropertyInfo {

    private static final Logger log = LoggerFactory.getLogger(EntityPropertyInfo.class);

    private static final Map<String, List<EntityPropertyInfo>> propertiesMap = new HashMap<>();

    static {
        List<EntityPropertyInfo> schemaProperties = new ArrayList<>();
        propertiesMap.put(TapSchema.SCHEMAS_TABLE, schemaProperties);
        schemaProperties.add(new EntityPropertyInfo(Schema.SCHEMA_NAME_KEY, String.class, false));
        schemaProperties.add(new EntityPropertyInfo(Schema.UTYPE_KEY, String.class, true));
        schemaProperties.add(new EntityPropertyInfo(Schema.DESCRIPTION_KEY, String.class, true));
        schemaProperties.add(new EntityPropertyInfo(Schema.SCHEMA_ID, Long.class, true, TapSchemaVersion.TAP_SCHEMA_1_IA2, TapSchemaVersion.TAP_SCHEMA_1_1_IA2));

        List<EntityPropertyInfo> tableProperties = new ArrayList<>();
        propertiesMap.put(TapSchema.TABLES_TABLE, tableProperties);
        tableProperties.add(new EntityPropertyInfo(Table.SCHEMA_NAME_KEY, String.class, false));
        tableProperties.add(new EntityPropertyInfo(Table.TABLE_NAME_KEY, String.class, false));
        tableProperties.add(new EntityPropertyInfo(Table.TABLE_TYPE_KEY, String.class, false));
        tableProperties.add(new EntityPropertyInfo(Table.UTYPE_KEY, String.class, true));
        tableProperties.add(new EntityPropertyInfo(Table.DESCRIPTION_KEY, String.class, true));
        tableProperties.add(new EntityPropertyInfo(Table.TABLE_INDEX, Integer.class, true, TapSchemaVersion.TAP_SCHEMA_1_1, TapSchemaVersion.TAP_SCHEMA_1_1_IA2));
        tableProperties.add(new EntityPropertyInfo(Table.TABLE_ID, Long.class, true, TapSchemaVersion.TAP_SCHEMA_1_IA2, TapSchemaVersion.TAP_SCHEMA_1_1_IA2));

        List<EntityPropertyInfo> columnProperties = new ArrayList<>();
        propertiesMap.put(TapSchema.COLUMNS_TABLE, columnProperties);
        columnProperties.add(new EntityPropertyInfo(Column.TABLE_NAME_KEY, String.class, false));
        columnProperties.add(new EntityPropertyInfo(Column.COLUMN_NAME_KEY, String.class, false));
        columnProperties.add(new EntityPropertyInfo(Column.DATATYPE_KEY, String.class, false));
        columnProperties.add(new EntityPropertyInfo(Column.ARRAYSIZE_KEY, Integer.class, false, TapSchemaVersion.TAP_SCHEMA_1_1, TapSchemaVersion.TAP_SCHEMA_1_1_IA2));
        columnProperties.add(new EntityPropertyInfo(Column.SIZE_KEY, Integer.class, false));
        columnProperties.add(new EntityPropertyInfo(Column.DESCRIPTION_KEY, String.class, true));
        columnProperties.add(new EntityPropertyInfo(Column.UTYPE_KEY, String.class, true));
        columnProperties.add(new EntityPropertyInfo(Column.UNIT_KEY, String.class, true));
        columnProperties.add(new EntityPropertyInfo(Column.UCD_KEY, String.class, true));
        columnProperties.add(new EntityPropertyInfo(Column.INDEXED_KEY, Boolean.class, false));
        columnProperties.add(new EntityPropertyInfo(Column.PRINCIPAL_KEY, Boolean.class, true));
        columnProperties.add(new EntityPropertyInfo(Column.STD_KEY, Boolean.class, true));
        columnProperties.add(new EntityPropertyInfo(Column.COLUMN_INDEX_KEY, Integer.class, true, TapSchemaVersion.TAP_SCHEMA_1_1, TapSchemaVersion.TAP_SCHEMA_1_1_IA2));
        columnProperties.add(new EntityPropertyInfo(Column.COLUMN_ID_KEY, Long.class, true, TapSchemaVersion.TAP_SCHEMA_1_IA2, TapSchemaVersion.TAP_SCHEMA_1_1_IA2));

        List<EntityPropertyInfo> keyProperties = new ArrayList<>();
        propertiesMap.put(TapSchema.KEYS_TABLE, keyProperties);
        keyProperties.add(new EntityPropertyInfo(Key.ID_KEY, String.class, true));
        keyProperties.add(new EntityPropertyInfo(Key.FROM_TABLE_KEY, String.class, false));
        keyProperties.add(new EntityPropertyInfo(Key.TARGET_TABLE_KEY, String.class, false));
        keyProperties.add(new EntityPropertyInfo(Key.DESCRIPTION_KEY, String.class, true));
        keyProperties.add(new EntityPropertyInfo(Key.UTYPE_KEY, String.class, true));
        keyProperties.add(new EntityPropertyInfo(Key.KEY_ID_KEY, Long.class, true, TapSchemaVersion.TAP_SCHEMA_1_IA2, TapSchemaVersion.TAP_SCHEMA_1_1_IA2));

        List<EntityPropertyInfo> keyColumnProperties = new ArrayList<>();
        propertiesMap.put(TapSchema.KEY_COLUMNS_TABLE, keyColumnProperties);
        keyColumnProperties.add(new EntityPropertyInfo(KeyColumn.KEY_ID_KEY, String.class, true));
        keyColumnProperties.add(new EntityPropertyInfo(KeyColumn.FROM_COLUMN_KEY, String.class, false));
        keyColumnProperties.add(new EntityPropertyInfo(KeyColumn.TARGET_COLUMN_KEY, String.class, false));
        keyColumnProperties.add(new EntityPropertyInfo(KeyColumn.KEY_COLUMN_ID_KEY, Long.class, true, TapSchemaVersion.TAP_SCHEMA_1_IA2, TapSchemaVersion.TAP_SCHEMA_1_1_IA2));
    }

    /**
     * Obtains all the {@link EntityPropertyInfo} for a given TAP_SCHEMA table
     * name.
     */
    public static List<EntityPropertyInfo> getEntityPropertiesInfo(String tapSchemaTable) {
        return propertiesMap.get(tapSchemaTable);
    }

    /**
     * Obtains the {@link EntityPropertyInfo} for a given TAP_SCHEMA table name
     * and a given column name.
     */
    public static EntityPropertyInfo getEntityPropertyInfo(String tapSchemaTable, String key) {
        for (EntityPropertyInfo propertyInfo : getEntityPropertiesInfo(tapSchemaTable)) {
            if (propertyInfo.getPropertyKey().equals(key)) {
                return propertyInfo;
            }
        }
        log.debug("property {} not found for {} table", key, tapSchemaTable);
        return null;
    }

    private final String propertyKey;
    private final Class type;
    private final int sqlType;
    private final boolean updatable;
    private final TapSchemaVersion[] versions;

    protected EntityPropertyInfo(String propertyKey, Class type, boolean updatable, TapSchemaVersion... versions) {
        this.propertyKey = propertyKey;
        this.type = type;

        if (type == String.class) {
            sqlType = Types.VARCHAR;
        } else if (type == Integer.class) {
            sqlType = Types.INTEGER;
        } else if (type == Long.class) {
            sqlType = Types.BIGINT;
        } else if (type == Boolean.class) {
            sqlType = Types.BIT;
        } else {
            throw new IllegalArgumentException("EntityPropertyInfo doesn't support class type " + type.getCanonicalName());
        }

        this.updatable = updatable;
        this.versions = versions;
    }

    /**
     * The name of the property (the column name).
     */
    public String getPropertyKey() {
        return propertyKey;
    }

    /**
     * The class type of the property value (the value stored in a
     * {@link EntityProperty}).
     */
    public Class getPropertyType() {
        return type;
    }

    /**
     * The {@link java.sql.Types} integer related to the class type of the
     * property value.
     */
    public int getSqlType() {
        return sqlType;
    }

    /**
     * Returns true if the user can update the value of this property.
     */
    public boolean isUpdatable() {
        return updatable;
    }

    /**
     * Returns all the TAP_SCHEMA versions that uses this property.
     */
    public TapSchemaVersion[] getVersions() {
        return versions;
    }

    /**
     * Returns true if the specified {@link TapSchemaVersion} uses this
     * property.
     */
    public boolean acceptVersion(TapSchemaVersion version) {
        if (versions == null || versions.length == 0) {
            return true;
        }
        for (TapSchemaVersion v : versions) {
            if (v == version) {
                return true;
            }
        }
        return false;
    }
}
