/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.oats.ia2.tapschemamanager.api;

import it.inaf.oats.ia2.tapschemamanager.api.contract.Column;
import it.inaf.oats.ia2.tapschemamanager.api.contract.Key;
import it.inaf.oats.ia2.tapschemamanager.api.contract.KeyColumn;
import it.inaf.oats.ia2.tapschemamanager.api.contract.Status;
import it.inaf.oats.ia2.tapschemamanager.api.contract.Table;
import it.inaf.oats.ia2.tapschemamanager.api.contract.TapSchema;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The main implementation of {@link Column}.
 * 
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class ColumnImpl extends ChildEntityImpl<Table> implements Column {

    private static final long serialVersionUID = 9175956487892235521L;
    private static final Logger log = LoggerFactory.getLogger(ColumnImpl.class);

    private boolean isPrimaryKey;

    private boolean foreignKeySearched;
    private Key foreignKey;

    private Table parentTable;

    private ColumnImpl() {
        // for serialization
        super();
    }

    protected ColumnImpl(DBWrapper dbWrapper, TapSchema tapSchema, Table table, String columnName, boolean indexed, boolean isPrimaryKey, String datatype, Integer size, Integer arraySize) {
        super(dbWrapper, tapSchema);

        this.isPrimaryKey = isPrimaryKey;

        parentTable = table;

        addProperty(TABLE_NAME_KEY, new FixedEntityProperty<>(table.getCompleteName()));
        addProperty(COLUMN_NAME_KEY, new FixedEntityProperty<>(columnName));
        addProperty(DATATYPE_KEY, new FixedEntityProperty<>(datatype));
        addProperty(SIZE_KEY, new FixedEntityProperty<>(size));
        addProperty(ARRAYSIZE_KEY, new FixedEntityProperty<>(arraySize));
        addProperty(INDEXED_KEY, new FixedEntityProperty<>(indexed));

        // Updatables
        addProperty(DESCRIPTION_KEY, new EditableProperty<String>());
        addProperty(UTYPE_KEY, new EditableProperty<String>());
        addProperty(UNIT_KEY, new EditableProperty<String>());
        addProperty(UCD_KEY, new EditableProperty<String>());
        addProperty(PRINCIPAL_KEY, new EditableProperty<>(isPrimaryKey, false));
        addProperty(STD_KEY, new EditableProperty<>(false, false));
        addProperty(COLUMN_INDEX_KEY, new EditableProperty<Integer>());
        addProperty(COLUMN_ID_KEY, new EditableProperty<Long>());

        setStatus(Status.LOADED);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Key getForeignKey() {
        if (!foreignKeySearched) { // lazy loading (but the foreignKey value can be null, so we use this boolean)

            String tableCompleteName = getParent().getCompleteName();
            String columnName = getValue(COLUMN_NAME_KEY, String.class);

            keysloop: // we want to loop on all schema keys, also hidden ones.
            for (Key key : ((TapSchemaImpl) tapSchema).getAllKeys()) {
                if (key.getFromTableCompleteName().equals(tableCompleteName)) {
                    for (KeyColumn keyColumn : key.getKeyColumns()) {
                        if (keyColumn.getFromColumn().equals(columnName)) {
                            foreignKey = key;
                            break keysloop;
                        }
                    }
                }
            }

            foreignKeySearched = true;
        }

        return foreignKey;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getTableCompleteName() {
        return getValue(TABLE_NAME_KEY, String.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getName() {
        return getValue(COLUMN_NAME_KEY, String.class);
    }

    @Override
    public boolean getIndexed() {
        return getValue(INDEXED_KEY, Boolean.class);
    }

    /**
     * {@inheritDoc}
     *
     * This information is not stored into the TAP_SCHEMA, so it will be
     * reloaded from the source schema each time.
     */
    @Override
    public boolean isPrimaryKey() {
        return isPrimaryKey;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getDatatype() {
        return getValue(DATATYPE_KEY, String.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Integer getArraySize() {
        return getValue(ARRAYSIZE_KEY, Integer.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Integer getSize() {
        return getValue(SIZE_KEY, Integer.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getDescription() {
        return getValue(DESCRIPTION_KEY, String.class);
    }

    @Override
    public void setDescription(String description) {
        setValue(DESCRIPTION_KEY, description);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getUtype() {
        return getValue(UTYPE_KEY, String.class);
    }

    @Override
    public void setUtype(String utype) {
        setValue(UTYPE_KEY, utype);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getUnit() {
        return getValue(UNIT_KEY, String.class);
    }

    @Override
    public void setUnit(String unit) {
        setValue(UNIT_KEY, unit);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getUCD() {
        return getValue(UCD_KEY, String.class);
    }

    @Override
    public void setUCD(String ucd) {
        setValue(UCD_KEY, ucd);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean getPrincipal() {
        return getValue(PRINCIPAL_KEY, Boolean.class);
    }

    @Override
    public void setPrincipal(boolean principal) {
        setValue(PRINCIPAL_KEY, principal);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean getStd() {
        return getValue(STD_KEY, Boolean.class);
    }

    @Override
    public void setStd(boolean std) {
        setValue(STD_KEY, std);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Integer getColumnIndex() {
        return getValue(COLUMN_INDEX_KEY, Integer.class);
    }

    @Override
    public void setColumnIndex(Integer columnIndex) {
        setValue(COLUMN_INDEX_KEY, columnIndex);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Long getColumnID() {
        return getValue(COLUMN_ID_KEY, Long.class);
    }

    @Override
    public void setColumnID(Long columnID) {
        setValue(COLUMN_ID_KEY, columnID);
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 29 * hash + Objects.hashCode(getTableCompleteName());
        hash = 29 * hash + Objects.hashCode(getName());
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ColumnImpl other = (ColumnImpl) obj;
        if (!Objects.equals(this.getTableCompleteName(), other.getTableCompleteName())) {
            return false;
        }
        if (!Objects.equals(this.getName(), other.getName())) {
            return false;
        }
        return true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Table getParent() {
        return parentTable;
    }
}
