/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.webapp;

import it.inaf.ia2.tsm.TapSchema;
import it.inaf.ia2.tsm.datalayer.DBBroker;
import it.inaf.ia2.tsm.datalayer.DBBrokerFactory;
import it.inaf.ia2.tsm.datalayer.DBWrapper;
import it.inaf.ia2.tsm.model.TapSchemaModel;
import it.inaf.ia2.tsm.model.TapSchemaModels;
import java.io.Serializable;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import javax.annotation.PostConstruct;
import javax.faces.application.FacesMessage;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.validator.ValidatorException;
import javax.inject.Inject;
import javax.inject.Named;
import org.apache.deltaspike.core.api.scope.WindowScoped;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
@Named("schemaSelection")
@WindowScoped
public class SchemaSelectionBean implements Serializable {
    
    private static final long serialVersionUID = -5745720427701334323L;
    private static final Logger LOG = LoggerFactory.getLogger(SchemaSelectionBean.class);
    
    @Inject
    TapSchemaEditingBean tapSchemaEditingBean;
    
    @Inject
    ConsistencyChecksBean consistencyChecksBean;
    
    private DBWrapper dbWrapper;
    
    private String selectedRadioOption;

    // For editing
    private List<String> allTAPSchemas;
    private String selectedTAPSchema;
    private String exposedSchemas;

    // For creation
    private String tapSchemaName;
    private List<String> versions;
    private String version;
    private List<String> allSchemas;
    private List<String> selectedSchemas;
    private boolean loading;
    private TapSchema loadedTapSchema;
    private String loadingError;
    
    @PostConstruct
    public void init() {
        selectedRadioOption = "edit";
        exposedSchemas = "";
        versions = new ArrayList<>();
        Iterator<TapSchemaModel> ite = TapSchemaModels.getIterator();
        while (ite.hasNext()) {
            TapSchemaModel model = ite.next();
            versions.add(model.getVersion());
        }
        Collections.sort(versions);
    }
    
    public void onPageLoad() {
        FacesContext fc = FacesContext.getCurrentInstance();
        final boolean ajaxRequest = fc.getPartialViewContext().isAjaxRequest();
        final boolean validationFailed = fc.isValidationFailed();
        
        if (!ajaxRequest && !validationFailed) {

            // Loading all schemas of the source database
            try {
                DBBroker broker = DBBrokerFactory.getDBBroker(dbWrapper.getSourceDataSourceWrapper());
                allSchemas = broker.getAllSchemaNames();
                setSelectedSchemas(new ArrayList<String>());
            } catch (SQLException e) {
                throw new RuntimeException(e);
            }

            // Loading all schemas of the TAP_SCHEMA database
            try {
                DBBroker broker = DBBrokerFactory.getDBBroker(dbWrapper.getTapSchemaDataSourceWrapper());
                allTAPSchemas = broker.getAllTAPSchemaNames(allSchemas);
                
                if (!allTAPSchemas.isEmpty()) {
                    this.selectedTAPSchema = allTAPSchemas.get(0);
                    loadExposedSchemas();
                }
            } catch (SQLException e) {
                throw new RuntimeException(e);
            }
        }
    }
    
    private void loadExposedSchemas() throws SQLException {
        DBBroker broker = DBBrokerFactory.getDBBroker(dbWrapper.getTapSchemaDataSourceWrapper());
        List<String> schemas = broker.getExposedSchemas(selectedTAPSchema);
        exposedSchemas = "";
        for (int i = 0; i < schemas.size(); i++) {
            exposedSchemas += schemas.get(i);
            if (i < schemas.size() - 1) {
                exposedSchemas += ", ";
            }
        }
    }
    
    public List<String> getAllTAPSchemas() {
        return allTAPSchemas;
    }
    
    public List<String> getAllSchemas() {
        return allSchemas;
    }
    
    public String getExposedSchemas() {
        return exposedSchemas;
    }
    
    public String getSelectedRadioOption() {
        return selectedRadioOption;
    }
    
    public void setSelectedRadioOption(String selectedRadioOption) {
        this.selectedRadioOption = selectedRadioOption;
    }
    
    public String getSelectedTAPSchema() {
        return selectedTAPSchema;
    }
    
    public void setSelectedTAPSchema(String selectedTAPSchema) {
        this.selectedTAPSchema = selectedTAPSchema;
    }
    
    public void selectedTAPSchemaChanged() {
        try {
            loadExposedSchemas();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }
    
    public List<String> getSelectedSchemas() {
        return selectedSchemas;
    }
    
    public void setSelectedSchemas(List<String> selectedSchemas) {
        this.selectedSchemas = selectedSchemas;
    }
    
    public String openLoaded() {
        if (loadedTapSchema.getConsistencyChecks().isInconsistent()) {
            consistencyChecksBean.setDbWrapper(dbWrapper);
            consistencyChecksBean.setTapSchema(loadedTapSchema);
            return "consistencyChecks.xhtml?faces-redirect=true";
        } else {
            tapSchemaEditingBean.setTapSchema(loadedTapSchema);
            return "tapSchemaEditing.xhtml?faces-redirect=true";
        }
    }
    
    public void edit() {
        
        loadedTapSchema = null;
        loading = true;
        loadingError = null;
        
        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    DBBroker broker = DBBrokerFactory.getDBBroker(dbWrapper.getTapSchemaDataSourceWrapper());
                    String version = broker.detectVersion(selectedTAPSchema);
                    loadedTapSchema = new TapSchema(version, dbWrapper, selectedTAPSchema, true);
                } catch (Throwable e) {
                    LOG.error("Exception caught", e);
                    loadingError = e.getMessage();
                    if (loadingError == null) {
                        loadingError = e.getClass().getCanonicalName();
                    }
                }
                loading = false;
            }
        }).start();
    }
    
    public void create() {
        
        loadedTapSchema = null;
        loading = true;
        loadingError = null;
        
        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    loadedTapSchema = new TapSchema(version, dbWrapper, tapSchemaName, false);
                    for (String schemaName : selectedSchemas) {
                        loadedTapSchema.addChild(schemaName);
                    }
                } catch (Throwable e) {
                    LOG.error("Exception caught", e);
                    if (e.getMessage() != null) {
                        loadingError = e.getMessage();
                    } else {
                        loadingError = e.getClass().getCanonicalName();
                        loadedTapSchema = null;
                    }
                }
                loading = false;
            }
        }).start();
    }
    
    public String getTapSchemaName() {
        return tapSchemaName;
    }
    
    public void setTapSchemaName(String tapSchemaName) {
        this.tapSchemaName = tapSchemaName;
    }
    
    public DBWrapper getDbWrapper() {
        return dbWrapper;
    }
    
    public void setDbWrapper(DBWrapper dbWrapper) {
        this.dbWrapper = dbWrapper;
    }
    
    public void validateTapSchemaName(FacesContext context, UIComponent inputComponent, Object value) {
        String textValue = (String) value;
        
        String validatorMessage = null;
        if (textValue == null || textValue.isEmpty()) {
            validatorMessage = "TAP_SCHEMA name is required";
        } else if (!textValue.matches("^[a-zA-Z0-9_[-]]*$")) {
            validatorMessage = "TAP_SCHEMA name has to be a valid table name";
        } else if (allSchemas.contains(textValue)) {
            validatorMessage = "Database already contains a schema with this name. Please choose another name";
        }
        
        if (validatorMessage != null) {
            throw new ValidatorException(new FacesMessage(validatorMessage));
        }
    }

    /**
     * This boolean is true when a TapSchema instance is loading.
     */
    public boolean isLoading() {
        return loading;
    }

    /**
     * This String is not null when an error happens while a TapSchema is
     * loading.
     */
    public String getLoadingError() {
        return loadingError;
    }
    
    public TapSchema getLoadedTapSchema() {
        return loadedTapSchema;
    }
    
    public List<String> getVersions() {
        return versions;
    }
    
    public String getVersion() {
        return version;
    }
    
    public void setVersion(String version) {
        this.version = version;
    }
}
