/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.webapp;

import it.inaf.ia2.tsm.datalayer.Credentials;
import it.inaf.ia2.tsm.datalayer.DBWrapper;
import java.io.IOException;
import java.io.Serializable;
import java.sql.SQLException;
import java.util.List;
import javax.annotation.PostConstruct;
import javax.faces.application.FacesMessage;
import javax.faces.context.FacesContext;
import javax.inject.Inject;
import javax.inject.Named;
import org.apache.deltaspike.core.api.scope.WindowScoped;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
@Named("credentialsInsertion")
@WindowScoped
public class CredentialsEditing implements Serializable {

    private static final long serialVersionUID = -2688980249773483198L;
    private static final Logger LOG = LoggerFactory.getLogger(CredentialsEditing.class);

    @Inject
    private User user;

    @Inject
    private ConfigurationData config;

    @Inject
    private SchemaSelectionBean schemaSelectionBean;

    private boolean credentialsDialogOpened;

    private Credentials sourceCredentials;
    private Credentials tapSchemaCredentials;
    private boolean separateCredentials;
    private int currentEditingRow;

    @PostConstruct
    public void init() {
        sourceCredentials = new Credentials();
        tapSchemaCredentials = new Credentials();
    }

    public List getSavedCredentials() {
        return user.getUserConfiguration().getCredentialsInfo();
    }

    public void editCredentials(Credentials credentials, int index) {
        this.sourceCredentials = credentials;
        separateCredentials = false;
        currentEditingRow = index;
    }

    public void editSeparateCredentials(SeparateCredentials sc, int index) {
        this.sourceCredentials = sc.getSourceCredentials();
        this.tapSchemaCredentials = sc.getTapSchemaCredentials();
        currentEditingRow = index;
        separateCredentials = true;
    }

    public void addNewCredentials() {
        separateCredentials = false;
        this.sourceCredentials = new Credentials();
        this.tapSchemaCredentials = new Credentials();
        currentEditingRow = getSavedCredentials().size();
    }

    public String loginWithSingleCredentials(Credentials credentials) {
        LOG.debug("Login with single credentials");
        return loginWithDBWrapper(new DBWrapper(credentials));
    }

    public String loginWithSeparatedCredentials(Credentials sourceCredentials, Credentials tapSchemaCredentials) {
        LOG.debug("Login with separated credentials");
        return loginWithDBWrapper(new DBWrapper(sourceCredentials, tapSchemaCredentials));
    }

    private String loginWithDBWrapper(DBWrapper dbWrapper) {

        try {
            dbWrapper.testConnections();
            schemaSelectionBean.setDbWrapper(dbWrapper);
            return "schemaSelection.xhtml?faces-redirect=true";
        } catch (SQLException e) {
            LOG.error("Exception caught", e);
            FacesContext.getCurrentInstance().addMessage("main", new FacesMessage("Connection error: " + e.getMessage()));
            return null;
        }
    }

    public void removeCredentials(int index) throws IOException {
        getSavedCredentials().remove(index);
        config.updateConfigurationFile();
    }

    public void saveCredentialsEdited() throws IOException {

        // If is editing existing, remove old for creating a new one
        if (currentEditingRow < getSavedCredentials().size()) {
            getSavedCredentials().remove(currentEditingRow);
        }

        if (separateCredentials) {
            SeparateCredentials sc = new SeparateCredentials(sourceCredentials, tapSchemaCredentials);
            getSavedCredentials().add(currentEditingRow, sc);
        } else {
            getSavedCredentials().add(currentEditingRow, sourceCredentials);
        }

        config.updateConfigurationFile();
    }

    public boolean isSeparateCredentials() {
        return separateCredentials;
    }

    public void setSeparateCredentials(boolean separateCredentials) {
        this.separateCredentials = separateCredentials;
    }

    public Credentials getSourceCredentials() {
        return sourceCredentials;
    }

    public void setSourceCredentials(Credentials sourceCredentials) {
        this.sourceCredentials = sourceCredentials;
    }

    public Credentials getTapSchemaCredentials() {
        return tapSchemaCredentials;
    }

    public void setTapSchemaCredentials(Credentials tapSchemaCredentials) {
        this.tapSchemaCredentials = tapSchemaCredentials;
    }

    public boolean isCredentialsDialogOpened() {
        return credentialsDialogOpened;
    }

    public void setCredentialsDialogOpened(boolean credentialsDialogOpened) {
        this.credentialsDialogOpened = credentialsDialogOpened;
    }
}
