/*
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2017 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.model;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import javax.xml.bind.JAXB;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class TapSchemaModels {

    // Map keys are TS versions
    private final static Map<String, TapSchemaModel> MODELS;

    private TapSchemaModels() {
    }

    static {

        try {
            Map<String, TapSchemaXMLModel> xmlModels = getXmlModels();

            MODELS = new HashMap<>();

            for (TapSchemaXMLModel xmlModel : xmlModels.values()) {
                TapSchemaModel model = new TapSchemaModel(xmlModel.getVersion(), xmlModel.getDescription());
                loadTapSchemaModel(model, xmlModel, xmlModels);
                MODELS.put(model.getVersion(), model);
            }
        } catch (IOException e) {
            throw new ExceptionInInitializerError(e);
        }
    }

    private static Map<String, TapSchemaXMLModel> getXmlModels() throws IOException {
        Map<String, TapSchemaXMLModel> xmlModels = new HashMap<>();

        for (String modelFile : Tasman.XML_MODEL_FILES) {
            try (InputStream in = TapSchemaModels.class.getClassLoader().getResourceAsStream(modelFile)) {
                TapSchemaXMLModel model = JAXB.unmarshal(in, TapSchemaXMLModel.class);
                xmlModels.put(model.getVersion(), model);
            }
        }

        return xmlModels;
    }

    private static void loadTapSchemaModel(TapSchemaModel model, TapSchemaXMLModel xmlModel, Map<String, TapSchemaXMLModel> xmlModels) {

        for (TableXMLModel tableXmlModel : xmlModel.getTables()) {
            String tableName = tableXmlModel.getName();
            TableModel tableModel = model.get(tableName);
            if (tableModel == null) {
                tableModel = new TableModel(tableName, tableXmlModel.getDescription());
            }
            for (PropertyXMLModel property : tableXmlModel.getAdd()) {
                tableModel.getProperties().put(property.getName(), new PropertyModel(property));
            }
            model.getTables().put(tableName, tableModel);
        }

        if (xmlModel.getExtendsFrom() != null) {
            TapSchemaXMLModel parentModel = xmlModels.get(xmlModel.getExtendsFrom());
            loadTapSchemaModel(model, parentModel, xmlModels);
        }
    }

    public static TapSchemaModel getTapSchemaModel(String version) {
        return MODELS.get(version);
    }

    public static Iterator<TapSchemaModel> getIterator() {
        return MODELS.values().iterator();
    }

    public static TableModel getTableModel(String tableName, String version) {
        return getTapSchemaModel(version).getTables().get(tableName);
    }
}
