/*
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2017 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.model;

import java.io.Serializable;
import java.util.List;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlTransient;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

/**
 * JAXB model for the structure and the behavior of a database column which is
 * created and edited by TASMAN.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class ColumnModel implements Serializable {

    private static final long serialVersionUID = -982004697900839996L;

    private String name;
    private String type;
    private Integer size;
    private boolean updatable;
    private boolean nullable;
    private boolean standard;
    private boolean mandatory;
    private String defaultValueString;
    private String loaderKey;
    private String description;
    private String ucd;
    private String utype;
    private String unit;
    private boolean principal;
    private List<String> enumValues;
    private Integer minValue;
    private Integer maxValue;

    public ColumnModel() {
        // default values
        updatable = true;
        nullable = true;
        // Currently this is used only for ObsCore
        mandatory = true;
    }

    /**
     * The column name
     */
    @XmlAttribute(name = "name")
    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    /**
     * The column type, in ADQL datatype format.
     */
    @XmlElement(name = "type")
    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    /**
     * The column size, if it can be set, null otherwise (for example VARCHAR
     * must have a size, INTEGER not).
     */
    @XmlElement(name = "size")
    public Integer getSize() {
        return size;
    }

    public void setSize(Integer size) {
        this.size = size;
    }

    /**
     * True if the user can edit the value, false otherwise.
     */
    @XmlElement(name = "updatable", defaultValue = "true")
    public boolean isUpdatable() {
        return updatable;
    }

    public void setUpdatable(boolean updatable) {
        this.updatable = updatable;
    }

    /**
     * True if the column can contain null values, false otherwise.
     */
    @XmlElement(name = "nullable", defaultValue = "true")
    public boolean isNullable() {
        return nullable;
    }

    public void setNullable(boolean nullable) {
        this.nullable = nullable;
    }

    /**
     * True if VO defines the column as a standard.
     */
    @XmlElement(name = "standard", defaultValue = "false")
    public boolean isStandard() {
        return standard;
    }

    public void setStandard(boolean standard) {
        this.standard = standard;
    }

    /**
     * Used for defining ObsCore mandatory or optional columns.
     */
    @XmlElement(name = "mandatory", defaultValue = "true")
    public boolean isMandatory() {
        return mandatory;
    }

    public void setMandatory(boolean mandatory) {
        this.mandatory = mandatory;
    }

    /**
     * Specifies the column default value.
     */
    @XmlElement(name = "default-value")
    public String getDefaultValueString() {
        return defaultValueString;
    }

    public void setDefaultValueString(String defaultValue) {
        this.defaultValueString = defaultValue;
    }

    /**
     * Key to use for retrieving the initial values of the column from the
     * {@code TapSchemaEntity} {@code metadata} field.
     */
    @XmlElement(name = "key")
    public String getLoaderKey() {
        return loaderKey;
    }

    public void setLoaderKey(String loaderMethod) {
        this.loaderKey = loaderMethod;
    }

    /**
     * Column description (used to fill TAP_SCHEMA schema description fields).
     */
    @XmlElement(name = "description")
    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * UCD for ObsCore columns.
     */
    @XmlElement(name = "ucd")
    public String getUcd() {
        return ucd;
    }

    public void setUcd(String ucd) {
        this.ucd = ucd;
    }

    /**
     * Utype for ObsCore columns.
     */
    @XmlElement(name = "utype")
    public String getUtype() {
        return utype;
    }

    public void setUtype(String utype) {
        this.utype = utype;
    }

    /**
     * VOUnit for ObsCore columns.
     */
    @XmlElement(name = "unit")
    public String getUnit() {
        return unit;
    }

    public void setUnit(String unit) {
        this.unit = unit;
    }

    /**
     * Defines if the columns is a principal one according to TAP standard.
     */
    @XmlElement(name = "principal", defaultValue = "false")
    public boolean isPrincipal() {
        return principal;
    }

    public void setPrincipal(boolean principal) {
        this.principal = principal;
    }

    /**
     * Used in ObsCore table for specifying an enum constraint.
     */
    @XmlElement(name = "enum-values", nillable = true, required = false)
    @XmlJavaTypeAdapter(CommaSeparatedListAdapter.class)
    public List<String> getEnumValues() {
        return enumValues;
    }

    public void setEnumValues(List<String> enumValues) {
        this.enumValues = enumValues;
    }

    /**
     * Defines the minimum value allowed for the column.
     */
    @XmlElement(name = "min", nillable = true, required = false)
    public Integer getMinValue() {
        return minValue;
    }

    public void setMinValue(Integer minValue) {
        this.minValue = minValue;
    }

    /**
     * Defines the maximum value allowed for the column.
     */
    @XmlElement(name = "max", nillable = true, required = false)
    public Integer getMaxValue() {
        return maxValue;
    }

    public void setMaxValue(Integer maxValue) {
        this.maxValue = maxValue;
    }

    /**
     * Retrieves the default value.
     */
    @XmlTransient
    public Object getDefaultValue() {
        if (defaultValueString == null) {
            return null;
        }
        return TypesMapping.parseTypedValue(defaultValueString, getType());
    }

    /**
     * Retrieves the Java type from the column datatype.
     */
    @XmlTransient
    public Class getJavaType() {
        return TypesMapping.getClassFromAdqlType(getType());
    }
}
