/*
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm;

import java.io.Serializable;

/**
 * Models a value stored in a TAP_SCHEMA column entity having a wrong value
 * according to the information read from the database metadata (or
 * information_schema).
 * <p>
 * Example: the structure of a table exposed by the TAP_SCHEMA has changed
 * setting a varchar column length from 255 to 50. In this case the
 * {@code arraysize} value will contain an inconsistent value and this has to be
 * detected by the consistency checking mechanism.
 *
 * @see it.inaf.ia2.tsm.ConsistencyChecks
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class InconsistentColumnProperty implements Serializable {

    private static final long serialVersionUID = -5145865322582594970L;

    private final ColumnHolder columnHolder;
    private final String key;
    private final Object currentValue;
    private final Object correctValue;

    /**
     * Default constructor.
     *
     * @param columnHolder the model representing the column owning the
     * inconsistent property.
     * @param key the name of the inconsistent property.
     * @param currentValue the (inconsistent) value stored into the TAP_SCHEMA.
     * @param correctValue the value read from the database metadata.
     */
    public InconsistentColumnProperty(ColumnHolder columnHolder, String key, Object currentValue, Object correctValue) {
        this.columnHolder = columnHolder;
        this.key = key;
        this.currentValue = currentValue;
        this.correctValue = correctValue;
    }

    /**
     * Returns the model representing the column owning the inconsistent
     * property.
     */
    public ColumnHolder getColumnHolder() {
        return columnHolder;
    }

    public String getTableCompleteName() {
        return String.format("%s.%s", columnHolder.getSchemaName(), columnHolder.getTableName());
    }

    public String getColumnName() {
        return columnHolder.getColumnName();
    }

    /**
     * Returns the name of the property containing the inconsistent value.
     */
    public String getKey() {
        return key;
    }

    /**
     * Returns the (inconsistent) value of the property stored into the
     * TAP_SCHEMA.
     */
    public Object getCurrentValue() {
        return currentValue;
    }

    /**
     * Returns the (correct) value of the property, as read from the database
     * metadata.
     */
    public Object getCorrectValue() {
        return correctValue;
    }
}
